/**
    NVT firmware update
    To do all-in-one firmware check and update
    @file       nvt_ivot_fw_update.c
    @ingroup
    @note
    Copyright   Novatek Microelectronics Corp. 2018.  All rights reserved.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as
    published by the Free Software Foundation.
*/

#include <cpu_func.h>
#include <common.h>
#include <cli.h>
#include <fs.h>
#include <u-boot/md5.h>
#include <malloc.h>
#include <nand.h>
#include <mmc.h>
#include <spi_flash.h>
#include <image.h>
#include <linux/delay.h>

#include <linux/libfdt.h>
#include <fdt.h>
#include <env.h>
#include <command.h>
#include <asm/nvt-common/nvt_types.h>
#include <asm/nvt-common/modelext/bin_info.h>
#include <asm/nvt-common/modelext/emb_partition_info.h>
#include <asm/nvt-common/shm_info.h>
#include <asm/nvt-common/nvt_common.h>
#include <asm/nvt-common/nvt_storage.h>
#include <asm/nvt-common/nvt_ivot_optee.h>
#include <asm/nvt-common/nvt_ivot_efuse_smc.h>
#include <asm/arch/IOAddress.h>
#include "nvt_ivot_pack.h"

#if defined(CONFIG_NVT_FW_UPDATE_LED)
#include <asm/arch/pwm.h>
#endif

#ifdef AB_MODIFIED
#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
extern void ab_fwup_led_toggle(void);
extern void ab_fwup_led_toggle_init(void);
extern void ab_fwup_led_toggle_deinit(void);
#endif
#endif


#define UPDATE_ALL_IN_ONE_RETRY_TIMES	3
#define CFG_MODEL_EXT_PARTITION 1

#define UINT32_SWAP(data)           (((((UINT32)(data)) & 0x000000FF) << 24) | \
                                     ((((UINT32)(data)) & 0x0000FF00) << 8) | \
                                     ((((UINT32)(data)) & 0x00FF0000) >> 8) | \
                                     ((((UINT32)(data)) & 0xFF000000) >> 24))   ///< Swap [31:24] with [7:0] and [23:16] with [15:8].

#ifdef AB_MODIFIED
#define MULTI_FW_PARTITION_OFFSET    (3)
#endif
static int nvt_flash_write(nvt_flash_type_t flash_type, u64 addr, u64 size, u64 part_off, u64 part_size, nvt_file_system_t fs_type)
{
    char command[128];
    int ret = 0;

	#if defined(CONFIG_NVT_FW_UPDATE_LED) && defined(CONFIG_NVT_PWM)
    pwm_setup_start(NVT_PWMLED, PWM_LED_ERASE, PWM_SIGNAL_TYPE);
    pwm_reload(NVT_PWMLED, PWM_LED_PROGRAM, PWM_SIGNAL_TYPE);
	#endif

	switch(flash_type) {
        case FLASHTYPE_NOR:
			sprintf(command, "sf erase 0x%llx +0x%llx", part_off, part_size);
			run_command(command, 0);
            if (fs_type == FS_RAW || fs_type == FS_SUPPORT) {
				sprintf(command, "sf write 0x%llx 0x%llx 0x%llx", addr, part_off, size);
			} else {
                printf("%s %s:flash type %d fs_type %d is not RAW or FS %s\r\n", ANSI_COLOR_RED, __func__, flash_type, fs_type, ANSI_COLOR_RESET);
                ret = -1;
            }
			run_command(command, 0);
			break;
		case FLASHTYPE_NAND:
			if (fs_type == FS_RAW) {
				sprintf(command, "nand erase 0x%llx 0x%llx", part_off, part_size);
				run_command(command, 0);
				sprintf(command, "nand write 0x%llx 0x%llx 0x%llx", addr, part_off, size);
				run_command(command, 0);
            } else if (fs_type == FS_SUPPORT) {
#if defined(_NVT_ROOTFS_TYPE_NAND_JFFS2_)
				sprintf(command, "nand erase clean 0x%llx 0x%llx", part_off, part_size);
#else
				sprintf(command, "nand erase 0x%llx 0x%llx", part_off, part_size);
#endif
				run_command(command, 0);
				sprintf(command, "nand info");
				run_command(command, 0);

#if defined(NVT_JFFS2_SUPPORT)
				printf("Update: %s %s\n", __func__, "JFFS2");
				sprintf(command, "nand write.jffs2 0x%llx 0x%llx 0x%llx", addr, part_off, size);
				printf("%s\n",command);
#else
				sprintf(command, "nand write.trimffs 0x%llx 0x%llx 0x%llx", addr, part_off, size);
				printf("%s\n",command);
#endif
				run_command(command, 0);
			} else {
                printf("%s %s:flash type %d fs_type %d is not RAW or FS %s\r\n", ANSI_COLOR_RED, __func__, flash_type, fs_type, ANSI_COLOR_RESET);
                ret = -1;
            }
			break;
		case FLASHTYPE_EMMC:
			if (fs_type == FS_RAW) {
				sprintf(command, "mmc write 0x%llx 0x%llx 0x%llx", addr, part_off, size);
				run_command(command, 0);
            } else if (fs_type == FS_SUPPORT) {
#if defined(CONFIG_NVT_LINUX_EMMC_BOOT)
				nvt_sparse_image_update(addr, part_off, size, part_size);
#else
				printf("%s: Please enable CONFIG_NVT_LINUX_EMMC_BOOT\n", __func__);
				ret = -1;
#endif
			} else {
                printf("%s %s:flash type %d fs_type %d is not RAW or FS %s\r\n", ANSI_COLOR_RED, __func__, flash_type, fs_type, ANSI_COLOR_RESET);
                ret = -1;
            }
			break;
		default:
			printf("%s %s:flash type %d is not support %s\r\n", ANSI_COLOR_RED, __func__, flash_type, ANSI_COLOR_RESET);
			ret = -1;
    }

	#if defined(CONFIG_NVT_FW_UPDATE_LED) && defined(CONFIG_NVT_PWM)
    pwm_close(NVT_PWMLED, 0);
    #endif

	return ret;
}

static int nvt_update_partitions(nvt_flash_type_t FlashType, u64 addr, u64 size, u64 part_off, u64 PartitionSize)
{
	u64 align_size = 0;
	u64 align_off = 0; // for emmc
	int ret = 0;

	if (size > (unsigned int)PartitionSize) {
        printf("%s partition size is too small (0x%08llx > 0x%08llx) %s\r\n", ANSI_COLOR_RED, size, PartitionSize, ANSI_COLOR_RESET);
        return -1;
    }

	switch(FlashType) {
	    case FLASHTYPE_NOR:
#if defined(CONFIG_NVT_SPI_NOR) || defined(CONFIG_NVT_SPI_NOR_NAND)
	        align_size = ALIGN_CEIL(size, nvt_get_flash_erasesize());
#else
			printf("%s: Please enable CONFIG_NVT_SPI_NOR\n", __func__);
			cli_loop();
#endif
			ret = nvt_flash_write(FlashType, addr, align_size, part_off, PartitionSize, FS_RAW);
			break;
		case FLASHTYPE_NAND:
#if defined(CONFIG_NVT_SPI_NAND) || defined(CONFIG_NVT_SPI_NOR_NAND)
			align_size = ALIGN_CEIL(size, nand_get_block_size());
#else
			printf("%s: Please enable CONFIG_NVT_SPI_NAND\n", __func__);
			cli_loop();
#endif
			ret = nvt_flash_write(FlashType, addr, align_size, part_off, PartitionSize, FS_RAW);
			break;
		case FLASHTYPE_EMMC:
			align_size = ALIGN_CEIL(size, MMC_MAX_BLOCK_LEN);
			/* Using block unit */
			align_size /= MMC_MAX_BLOCK_LEN;
			align_off = ALIGN_CEIL(part_off, MMC_MAX_BLOCK_LEN);
			/* Using block unit */
			align_off /= MMC_MAX_BLOCK_LEN;

			ret = nvt_flash_write(FlashType, addr, align_size, align_off, PartitionSize, FS_RAW);
	        break;
	    default:
	        printf("%s %s:flash type %d is not support %s\r\n", ANSI_COLOR_RED, __func__, FlashType, ANSI_COLOR_RESET);
			return -1;
	}
	return ret;
}

static int nvt_read_partitions(nvt_flash_type_t FlashType, u64 addr, u64 size, u64 part_off, u32 pat_id, unsigned short embtype)
{
	char command[128];
	u64 align_size = 0;
	u64 align_off = 0; // for emmc

	memset(command, 0, sizeof(command));

	switch(FlashType) {
	case FLASHTYPE_NOR:
		sprintf(command, "sf read 0x%llx 0x%llx 0x%llx", addr, part_off, size);
		run_command(command, 0);
		break;
	case FLASHTYPE_NAND:
		sprintf(command, "nand read 0x%llx 0x%llx 0x%llx", addr, part_off, size);
		run_command(command, 0);
		break;
	case FLASHTYPE_EMMC:
		align_off = ALIGN_CEIL(part_off, MMC_MAX_BLOCK_LEN);
		/* Using block unit */
		align_off /= MMC_MAX_BLOCK_LEN;
		align_size = ALIGN_CEIL(size, MMC_MAX_BLOCK_LEN);
		/* Using block unit */
		align_size /= MMC_MAX_BLOCK_LEN;
		sprintf(command, "mmc read 0x%llx 0x%llx 0x%llx", addr, align_off, align_size);
		run_command(command, 0);
		break;
	default:
		printf("%s %s:flash type %d is not support %s\r\n", ANSI_COLOR_RED, __func__, FlashType, ANSI_COLOR_RESET);
		return -1;
	}

	return 0;
}

static int nvt_update_fs_partition(nvt_flash_type_t FlashType, u64 addr, u64 size, u64 part_off, u64 part_size, EMB_PARTITION *pEmb)
{
	u64 align_size = 0;
    u64 align_off = 0; // for emmc
	int ret = 0;

#ifdef CONFIG_NVT_BIN_CHKSUM_SUPPORT
	// Skip nvt head info
	addr += 64;
	size -= 64;
#endif

	if (size > part_size) {
		printf("%s partition size is too small (0x%08llx > 0x%08llx) %s\r\n", ANSI_COLOR_RED, size, part_size, ANSI_COLOR_RESET);
		return -1;
	}

	switch(FlashType) {
    case FLASHTYPE_NOR:
#if defined(NVT_LINUX_RAMDISK_SUPPORT)
		size = ALIGN_CEIL(size, 4096);
#endif
		ret = nvt_flash_write(FlashType, addr, size, part_off, part_size, FS_SUPPORT);
		break;
    case FLASHTYPE_NAND:
		ret = nvt_flash_write(FlashType, addr, size, part_off, part_size, FS_SUPPORT);
        break;
    case FLASHTYPE_EMMC:
#if defined(NVT_LINUX_RAMDISK_SUPPORT) || defined(NVT_SQUASH_SUPPORT)
		if (pEmb->EmbType == EMBTYPE_ROOTFS && pEmb->OrderIdx == 0) {
			align_size = ALIGN_CEIL(size, MMC_MAX_BLOCK_LEN);
			/* Using block unit */
			align_size /= MMC_MAX_BLOCK_LEN;
			align_off = ALIGN_CEIL(part_off, MMC_MAX_BLOCK_LEN);
			/* Using block unit */
			align_off /= MMC_MAX_BLOCK_LEN;
			/* Ramdisk update for rootfs partition idx 0 */
			ret = nvt_flash_write(FlashType, addr,  align_size, align_off, part_size, FS_RAW);
		} else {
			/* Other partition is ext4 */
			printf("Update: %s %s size=0x%08llx\n", __func__, "EXT4 or FAT", size);
			ret = nvt_flash_write(FlashType, addr,  size, part_off, part_size, FS_SUPPORT);
		}
#elif defined(NVT_EXT4_SUPPORT)
		printf("Update: %s %s size=0x%08x\n", __func__, "EXT4 or FAT", size);
		ret = nvt_flash_write(FlashType, addr,  size, part_off, part_size, FS_SUPPORT);
#else
		printf("%s: No corresponding file system in FLASHTYPE_EMMC\n", __func__);
        return -1;
#endif
        break;
    default:
		printf("%s %s:flash type %d is not support %s\r\n", ANSI_COLOR_RED, __func__, FlashType, ANSI_COLOR_RESET);
        return -1;
    }

	return ret;
}

static int nvt_read_fs_partition(nvt_flash_type_t FlashType, u64 addr, u64 size, u64 part_off, u64 part_size, u32 pat_id, EMB_PARTITION *pEmb)
{
	char command[128];
	int ret = 0;

	memset(command, 0, sizeof(command));
#ifdef CONFIG_NVT_BIN_CHKSUM_SUPPORT
	// Skip nvt head info
	addr += 64;
	size -= 64;
#endif
	switch(FlashType) {
        case FLASHTYPE_NOR:
			sprintf(command, "sf read 0x%llx 0x%llx 0x%llx", addr, part_off, size);
			run_command(command, 0);
			break;
		case FLASHTYPE_NAND:
			sprintf(command, "nand read 0x%llx 0x%llx 0x%llx", addr, part_off, size);
			run_command(command, 0);
			break;
		case FLASHTYPE_EMMC:
			printf("Read: %s %s size=0x%08llx\n", __func__, "EXT4 or FAT", size);
#if defined(CONFIG_NVT_LINUX_EMMC_BOOT)
			nvt_sparse_image_readback(addr, part_off, size, part_size);
#else
			printf("%s: device emmc is found\n", __func__);
#endif
            break;
        default:
            printf("%s %s:flash type %d is not support %s\r\n", ANSI_COLOR_RED, __func__, FlashType, ANSI_COLOR_RESET);
            return -1;
    }

#if defined(NVT_UBIFS_SUPPORT) && (defined(NVT_LINUX_RAMDISK_SUPPORT) && (defined(CONFIG_NVT_SPI_NAND)  || defined(CONFIG_NVT_SPI_NOR_NAND)))
	/*test ubifs*/
	printf("Read: %s %s\n", __func__, "UBIFS");
	if(pEmb->EmbType == EMBTYPE_ROOTFS){

		#if (defined(NVT_LINUX_RAMDISK_SUPPORT) && (defined(CONFIG_NVT_SPI_NAND) || defined(CONFIG_NVT_SPI_NOR_NAND)))
			if (pEmb->OrderIdx == 0){
				/*rootfs0 is ramdisk no need to check ubifs*/
				return ret;
			}
		#endif
		sprintf(command, "ubi part rootfs%u", pEmb->OrderIdx);
		run_command(command, 0);
		sprintf(command, "ubifsmount ubi:rootfs");
		run_command(command, 0);
	}
	else if(pEmb->EmbType == EMBTYPE_APP){
		sprintf(command, "ubi part app");
		run_command(command, 0);
		sprintf(command, "ubifsmount ubi:app");
		run_command(command, 0);
	}
	else{
		printk("partition type%x , not support UBIFS\n",pEmb->EmbType);
		return -1;
	}

	sprintf(command, "ubifsls");
	run_command(command, 0);
#elif defined(NVT_SQUASH_SUPPORT)
	printf("Read: %s %s\n", __func__, "SQUASHFS");
#elif defined(NVT_EXT4_SUPPORT)
	printf("Read: %s %s\n", __func__, "EXT4");
#else
	printf("Read: %s\n", __func__);
#endif

	return ret;
}

UINT32 MemCheck_CalcCheckSum16Bit(ulong uiAddr,UINT32 uiLen)
{
	UINT32 i,uiSum = 0;
	UINT16 *puiValue = (UINT16 *)uiAddr;

	for (i=0; i<(uiLen >> 1); i++)
	{
	uiSum += (*(puiValue + i) + i);
	}

	uiSum &= 0x0000FFFF;

	return uiSum;
}

static int nvt_chk_last_ebr(ulong ebr_addr, u64 ebr_part_offset, EMB_PARTITION *pEmb)
{
	unsigned char *buf;
	u32 disk_capacity, new_size, old_size, part_addr;
	int i, last_part_idx, partition_number, last_ebr_idx, first_mbr_idx;

	//Find the index of the last pstore/rootfs partition
	if(pEmb==NULL)
	{
		printf("failed to get current partition resource.\r\n");
		return CMD_RET_FAILURE;
	}

	last_part_idx = -1;
	for(i = (EMB_PARTITION_INFO_COUNT - 1) ; i >= 0 ; i--)
	{
		if(pEmb[i].EmbType == EMBTYPE_UNKNOWN)
			continue;
		else if(pEmb[i].EmbType == EMBTYPE_ROOTFS || pEmb[i].EmbType == EMBTYPE_ROOTFSL){
			if(pEmb[i].PartitionSize != 0){
				printf("size of last partition is not 0, no need to fix\r\n");
				return CMD_RET_SUCCESS;
			}
			else{
				last_part_idx = i;
			}
			break;
		}
		else{
			printf("last partition is of type %d, not rootfs or pstore, no need to fix\r\n", pEmb[i].EmbType);
			return CMD_RET_SUCCESS;
		}
	}

	if(last_part_idx == -1){
		printf("no rootfs,pstore partition, no need to fix\r\n");
		return CMD_RET_SUCCESS;
	}

	//Find the index of the last ebr
	last_ebr_idx = -1;
	for(i = (EMB_PARTITION_INFO_COUNT - 1) ; i >= 0 ; i--)
		if(pEmb[i].EmbType == EMBTYPE_MBR){
			last_ebr_idx = i;
			break;
		}

	if(last_ebr_idx == -1){
		printf("fail to get last ebr's index\r\n");
		return CMD_RET_FAILURE;
	}

	//Find the index of mbr
	first_mbr_idx = -1;
	for(i = 0 ; i < EMB_PARTITION_INFO_COUNT ; i++)
		if(pEmb[i].EmbType == EMBTYPE_MBR){
			first_mbr_idx = i;
			break;
		}

	if(first_mbr_idx == -1){
		printf("fail to get 1st mbr's index\r\n");
		return CMD_RET_FAILURE;
	}

	//If ebr from argument is neither 1st mbr nor last ebr, ignore it
	if(ebr_part_offset != pEmb[first_mbr_idx].PartitionOffset && ebr_part_offset != pEmb[last_ebr_idx].PartitionOffset)
		return CMD_RET_SUCCESS;

	//Get pstore/rootfs partition number
	partition_number = 0;
	for(i = 0 ; i < EMB_PARTITION_INFO_COUNT ; i++)
		if(pEmb[i].EmbType == EMBTYPE_ROOTFS || pEmb[i].EmbType == EMBTYPE_ROOTFSL)
			partition_number++;

	//If the last partition ls logical, make sure there is a EBR partition ahead
	if(partition_number >= 4){
		if((last_part_idx - 1) != last_ebr_idx){
			printf("last ebr index(%d) is not right ahead of the last mbr ebr index(%d)\r\n", last_ebr_idx, last_part_idx);
			return CMD_RET_FAILURE;
		}
	}
	//rootfs,pstore number < 4, there should be no ebr, so last_ebr_idx should eb equal to first_mbr_idx
	else if(first_mbr_idx != last_ebr_idx){
		printf("only %d rootfs,pstore partions, but mbr index(%d) != last ebr index(%d)\r\n", partition_number, first_mbr_idx, last_ebr_idx);
		return CMD_RET_FAILURE;
	}

	//Check mbr/ebr signature
	buf = (unsigned char*)ebr_addr;
	if(buf[510] != 0x55 || buf[511] != 0xAA){
		printf("invalid mbr ebr signature 0x%x 0x%x, they should be 0x55 0xAA\r\n", (unsigned int)buf[510], (unsigned int)buf[511]);
		return CMD_RET_FAILURE;
	}
#ifdef CONFIG_NVT_IVOT_EMMC
	//Get emmc's max capacity
	struct mmc *mmc = find_mmc_device(CONFIG_NVT_IVOT_EMMC);
	if (!mmc)
		return CMD_RET_FAILURE;

	disk_capacity = mmc->capacity;
	printf("emmc capacity is %d sectors\r\n", disk_capacity);
	if(disk_capacity == 0){
		printf("fail to get emmc's capacity\r\n");
		return CMD_RET_FAILURE;
	}
#endif
	//Fix MBR's size field
	if(ebr_part_offset == pEmb[first_mbr_idx].PartitionOffset){
		//Fix primary partition's size
		if(partition_number < 4){
			new_size = disk_capacity - (u32)(pEmb[last_part_idx].PartitionOffset/MMC_MAX_BLOCK_LEN);
			buf = (unsigned char*)(ebr_addr + 446 + ((partition_number - 1) * 16) + 12);
			old_size = (buf[0] | (buf[1] << 8) | (buf[2] << 16) | (buf[3] << 24));
			printf("last primary partition old size = %d ; new size = %d\r\n", old_size, new_size);
			buf[0] = (new_size & 0x0FF);
			buf[1] = ((new_size & 0x0FF00) >> 8);
			buf[2] = ((new_size & 0x0FF0000) >> 16);
			buf[3] = ((new_size & 0xFF000000) >> 24);
		}
		//Fix extend partition's size
		else{
			buf = (unsigned char*)(ebr_addr + 494 + 8);
			part_addr = (buf[0] | (buf[1] << 8) | (buf[2] << 16) | (buf[3] << 24));
			new_size = disk_capacity - part_addr;
			buf = (unsigned char*)(ebr_addr + 494 + 12);
			old_size = (buf[0] | (buf[1] << 8) | (buf[2] << 16) | (buf[3] << 24));
			printf("extended partition old size = %d ; new size = %d\r\n", old_size, new_size);
			buf[0] = (new_size & 0x0FF);
			buf[1] = ((new_size & 0x0FF00) >> 8);
			buf[2] = ((new_size & 0x0FF0000) >> 16);
			buf[3] = ((new_size & 0xFF000000) >> 24);
		}
	}
	//Fix EBR's size field
	else{
		new_size = disk_capacity - (u32)(pEmb[last_part_idx].PartitionOffset/MMC_MAX_BLOCK_LEN);
		buf = (unsigned char*)(ebr_addr + 446 + 12);
		old_size = (buf[0] | (buf[1] << 8) | (buf[2] << 16) | (buf[3] << 24));
		printf("last ebr old size = %d ; new size = %d\r\n", old_size, new_size);
		buf[0] = (new_size & 0x0FF);
		buf[1] = ((new_size & 0x0FF00) >> 8);
		buf[2] = ((new_size & 0x0FF0000) >> 16);
		buf[3] = ((new_size & 0xFF000000) >> 24);
	}
	return CMD_RET_SUCCESS;
}

static int nvt_chk_app(ulong addr, unsigned int size, unsigned int pat_id){

	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
	{
		printf("app pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}
	return 0;

}

static int nvt_chk_loader(ulong addr, unsigned int size, unsigned int pat_id)
{
	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
	{
		printf("loader pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}
	return 0;
}

static int nvt_chk_modelext(ulong addr, unsigned int size, unsigned int pat_id)
{
	int  ret;

#if defined(CONFIG_ENCRYPT_DTS)
	#if defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
	if (is_secure_enable() != 0) {
		if (MemCheck_CalcCheckSum16Bit(addr, size) != 0) {
			printf("fdt pat%d, res check sum fail.\r\n", pat_id);
			return -1;
		}
	}
	#elif defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
	NVT_SMC_EFUSE_DATA efuse_data = {0};
	efuse_data.cmd = NVT_SMC_EFUSE_IS_SECURE;
	if (nvt_ivot_optee_efuse_operation(&efuse_data) != 0) {
		if (MemCheck_CalcCheckSum16Bit(addr, size) != 0) {
			printf("fdt pat%d, res check sum fail.\r\n", pat_id);
			return -1;
		}
	}
	#endif /* CONFIG_NVT_IVOT_SECBOOT_SUPPORT */
	nvt_dbg(IND, "Verify encrypted dts PASS\r\n");
	return 0;
#endif /* CONFIG_ENCRYPT_DTS */

	ret = nvt_check_isfdt(addr);
	if (ret < 0)
		printf("fdt pat%d, res check sum fail.\r\n", pat_id);

	return ret;
}

static int nvt_chk_rtos(ulong addr, unsigned int size, unsigned int pat_id)
{
	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
	{
		printf("ecos pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}
	return 0;
}

static int nvt_chk_atf(ulong addr, unsigned int size, unsigned int pat_id)
{
	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0) {
		printf("atf pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}
	return 0;
}

static int nvt_chk_teeos(ulong addr, unsigned int size, unsigned int pat_id){


	#ifdef CONFIG_NVT_IVOT_OPTEE_SUPPORT
	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
	{
		printf("teeos pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}
	#else
		printf("please enable CONFIG_NVT_IVOT_OPTEE_SUPPORT\n");
		return -1;
	#endif
	return 0;
}

static int nvt_chk_ai(ulong addr, unsigned int size, unsigned int pat_id)
{
#if 0
	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
	{
		printf("ai pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}

	return 0;
#else
	// ai model context is user defined.
	printf("skip checksum of ai model.\n");

	return 0;
#endif
}

static int nvt_chk_uboot(ulong addr, unsigned int size, unsigned int pat_id)
{
#ifdef CONFIG_ENCRYPT_UBOOT
	unsigned int is_secure = 0;
	#if (defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT))
	NVT_SMC_EFUSE_DATA efuse_data = {0};
	efuse_data.cmd = NVT_SMC_EFUSE_IS_SECURE;
	is_secure = nvt_ivot_optee_efuse_operation(&efuse_data);
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
	is_secure = is_secure_enable();
	#endif
#endif

#ifdef CONFIG_NVT_BIN_CHKSUM_SUPPORT
	extern ulong __image_copy_start;
	extern HEADINFO gHeadInfo;
	ulong tag_offset;
	char *tag;
	NVTPACK_BFC_HDR *pbfc = (NVTPACK_BFC_HDR *)addr;
	gHeadInfo.CodeEntry = (ulong)&__image_copy_start;
	tag_offset = (unsigned long)gHeadInfo.BinInfo_1 - (unsigned long)gHeadInfo.CodeEntry;
	tag = (char*)(addr + tag_offset);

#ifdef CONFIG_ENCRYPT_UBOOT
	/* This is for compression check */
	if(is_secure == 0)
	{
		if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
		{
			printf("uboot pat%d, check sum fail.\r\n",pat_id);
			return -1;
		}
		if(pbfc->uiFourCC == MAKEFOURCC('B', 'C', 'L', '1')) {
			// decode 0x200 size for get BinInfo_1
			const int preload_size = 0x200;
			// enlarge 10 times to avoid buffer overwrite on decompressing
			const int tmp_size = preload_size*10;
			unsigned char *p_tmp = (unsigned char *)malloc(tmp_size);
			if (p_tmp == NULL) {
				printf("no mem to decode uboot.lz, req size = 0x%08X\n", tmp_size);
				return -1;
			}

			memset(p_tmp, '\0', tmp_size);

			if((cpu_to_be32(pbfc->uiAlgorithm) & 0xFF ) == 11) {
				printf("uboot.lzma used, skip tag check.\r\n");
			} else {
				lz_uncompress((unsigned char *)(addr + sizeof(NVTPACK_BFC_HDR)),
						(unsigned char *)p_tmp,
						preload_size);
				tag = (char*)(p_tmp + tag_offset);
				if(strncmp(tag, gHeadInfo.BinInfo_1, 8) !=0 ) {
					printf("uboot pat%d, tag not match %8s(expect) != %8s(bin).\r\n"
					    ,pat_id ,gHeadInfo.BinInfo_1, tag);
					tag = NULL;
					free((void*)p_tmp);
					return -1;
				}
			}
			free((void*)p_tmp);
		}
	}
	else
	{
		if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
		{
			printf("uboot pat%d, res check sum fail.\r\n",pat_id);
			return -1;
		}
	}
#else
	/* This is for compression check */
	if(MemCheck_CalcCheckSum16Bit(addr, size)!=0)
	{
		printf("uboot pat%d, check sum fail.\r\n",pat_id);
		return -1;
	}
	if(pbfc->uiFourCC == MAKEFOURCC('B', 'C', 'L', '1')) {
		// decode 0x200 size for get BinInfo_1
		const int preload_size = 0x200;
		// enlarge 10 times to avoid buffer overwrite on decompressing
		const int tmp_size = preload_size*10;
		unsigned char *p_tmp = (unsigned char *)malloc(tmp_size);
		if (p_tmp == NULL) {
			printf("no mem to decode uboot.lz, req size = 0x%08X\n", tmp_size);
			return -1;
		}

		memset(p_tmp, '\0', tmp_size);

		if((cpu_to_be32(pbfc->uiAlgorithm) & 0xFF ) == 11) {
			printf("uboot.lzma used, skip tag check.\r\n");
		} else {
			lz_uncompress((unsigned char *)(addr + sizeof(NVTPACK_BFC_HDR)),
					(unsigned char *)p_tmp,
					preload_size);
			tag = (char*)(p_tmp + tag_offset);
			if(strncmp(tag, gHeadInfo.BinInfo_1, 8) !=0 ) {
				printf("uboot pat%d, tag not match %8s(expect) != %8s(bin).\r\n"
					,pat_id ,gHeadInfo.BinInfo_1, tag);
				tag = NULL;
				free((void*)p_tmp);
				return -1;
			}
		}
		free((void*)p_tmp);
	}
#endif /* CONFIG_ENCRYPT_UBOOT */
#endif /* CONFIG_NVT_BIN_CHKSUM_SUPPORT */
	return 0;
}

static int nvt_chk_linux(ulong addr, unsigned int size, unsigned int pat_id)
{
#ifdef CONFIG_ENCRYPT_KERNEL
	unsigned int is_secure = 0;
	#if (defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT))
	NVT_SMC_EFUSE_DATA efuse_data = {0};
	efuse_data.cmd = NVT_SMC_EFUSE_IS_SECURE;
	is_secure = nvt_ivot_optee_efuse_operation(&efuse_data);
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
	is_secure = is_secure_enable();
	#endif

	if(is_secure == 0)
	{
		if(*(unsigned int *)(addr) == MAKEFOURCC('B', 'C', 'L', '1')) {
			if(MemCheck_CalcCheckSum16Bit(addr, size) !=0) {
				printf("uImage checksum fail\n");
				return -1;
			}
		} else if(image_check_dcrc((const image_header_t*)(addr))!=1) {
			printf("linux-kernel pat%d, res check sum fail.\r\n",pat_id);
			return -1;
		}
	}
	else
	{
		printf("%s uImage is encryped , check by checksum\r\n",ANSI_COLOR_YELLOW);
		if(MemCheck_CalcCheckSum16Bit(addr, size) !=0)
		{
			printf("uImage checksum fail\n");
			return -1;
		}

	}
#else
	if(*(unsigned int *)(addr) == MAKEFOURCC('B', 'C', 'L', '1')) {
		if(MemCheck_CalcCheckSum16Bit(addr, size) !=0) {
			printf("uImage checksum fail\n");
			return -1;
		}
	} else if(image_check_dcrc((const image_header_t*)(addr))!=1) {
		printf("linux-kernel pat%d, res check sum fail.\r\n",pat_id);
		return -1;
	}
#endif /* CONFIG_ENCRYPT_KERNEL */

	return 0;
}

static int nvt_chk_rootfs(ulong addr, unsigned int size, unsigned int pat_id)
{
#ifdef CONFIG_NVT_BIN_CHKSUM_SUPPORT
	ulong uiContextBuf = addr;
	if(*(unsigned int*)uiContextBuf == MAKEFOURCC('C','K','S','M'))
	{
		NVTPACK_CHKSUM_HDR* pHdr = (NVTPACK_CHKSUM_HDR*)uiContextBuf;
		if(pHdr->uiVersion != NVTPACK_CHKSUM_HDR_VERSION)
		{
			printf("Wrong HEADER_CHKSUM_VERSION %08X(uboot) %08X(root-fs).\r\n",NVTPACK_CHKSUM_HDR_VERSION,pHdr->uiVersion);
			return -1;
		}
		UINT32 uiLen = pHdr->uiDataOffset + pHdr->uiDataSize + pHdr->uiPaddingSize;
		if(MemCheck_CalcCheckSum16Bit(uiContextBuf,uiLen)!=0)
		{
			printf("pat%d, res check sum fail.\r\n",pat_id);
			return -1;
		}
	}
	else
	{
#ifdef CONFIG_ENCRYPT_ROOTFS
		printf("%s rootfs is encryped , check by checksum\r\n",ANSI_COLOR_YELLOW);
		if(MemCheck_CalcCheckSum16Bit(addr, size) !=0)
		{
			printf("rootfs checksum fail\n");
			return -1;
		}
#else
		printf("root-fs has no CKSM header\r\n");
		return -1;
#endif
	}
#endif /* CONFIG_NVT_BIN_CHKSUM_SUPPORT */
	return 0;
}

int nvt_chk_all_in_one_valid(unsigned short EmbType, unsigned int addr, unsigned int size, unsigned int id)
{
	switch(EmbType)
	{
		case EMBTYPE_LOADER:
			if(nvt_chk_loader((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_FDT:
			if(nvt_chk_modelext((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_UBOOT:
			if(nvt_chk_uboot((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_ATF:
			if(nvt_chk_atf((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_LINUX:
			if(nvt_chk_linux((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_ROOTFS:
        case EMBTYPE_ROOTFS1:
        case EMBTYPE_RES:
			if(nvt_chk_rootfs((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_RTOS:
			if(nvt_chk_rtos((ulong)addr,size,id)!=0)
				return -1;
			break;
		case EMBTYPE_APP:
			if(nvt_chk_app((ulong)addr,size,id)!=0){
				return -1;
			}
			break;
		case EMBTYPE_TEEOS:
			if(nvt_chk_teeos((ulong)addr,size,id)!=0){
				return -1;
			}
			break;
		case EMBTYPE_AI:
			if(nvt_chk_ai((ulong)addr,size,id)!=0){
				return -1;
			}
			break;
		case EMBTYPE_FDTFAST:
			/* fdtfast is always non secure */
			if (nvt_check_isfdt((ulong)addr)!= 0)
				return -1;
			break;
		default:
			break;
	}

	return 0;
}

static int nvt_on_partition_enum_sanity(unsigned int id, NVTPACK_MEM* p_mem, void* p_user_data)
{
	int ret = 0;
	#if defined(CONFIG_NVT_LINUX_SPINAND_BOOT) || defined(CONFIG_NVT_LINUX_SPINORNAND_BOOT) || defined(CONFIG_NVT_LINUX_SPINOR_BOOT) || defined(CONFIG_NVT_LINUX_EMMC_BOOT)
	EMB_PARTITION* pEmb = (EMB_PARTITION*)p_user_data;
	#if defined(CONFIG_NVT_LINUX_SPINAND_BOOT) || defined(CONFIG_NVT_LINUX_SPINORNAND_BOOT)
	u64 PartitionSize = pEmb[id].PartitionSize;
	#elif defined(CONFIG_NVT_IVOT_EMMC)
	u64 PartitionSize = pEmb[id].PartitionSize;
	#elif defined(CONFIG_NVT_LINUX_SPINOR_BOOT)
	u64 PartitionSize = pEmb[id].PartitionSize;
	#endif
	unsigned int size = p_mem->len;

	#ifdef AB_MODIFIED
	#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
	ab_fwup_led_toggle();
	#endif
	#endif

#if defined(CONFIG_NVT_LINUX_SPINAND_BOOT)
	uint32_t block_size = nand_get_block_size();
	if( block_size == 0){
		printf("%s [WARNING] nand_get_block_size failed.%s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
		return -1;
	}

	unsigned int reserve_blocks = rounddown(PartitionSize - size, block_size) / block_size;
	if(reserve_blocks < 5 && size > 0x100000)
	{
		printf("%s [WARNING] Partition (%d) reserve size is too small, only %d blocks are reserved to cover bad blocks.%s\r\n", ANSI_COLOR_YELLOW, id, reserve_blocks, ANSI_COLOR_RESET);
	}
#endif


	#ifdef CONFIG_NVT_BIN_CHKSUM_SUPPORT
	// Skip rootfs nvt head info
	#ifdef AB_MODIFIED
	if ((pEmb[id].EmbType == EMBTYPE_ROOTFS) || (pEmb[id].EmbType == EMBTYPE_USER) || (pEmb[id].EmbType == EMBTYPE_USERRAW)
	        || (pEmb[id].EmbType == EMBTYPE_USER_1) || (pEmb[id].EmbType == EMBTYPE_USER_2) || (pEmb[id].EmbType == EMBTYPE_USER_3) ) {
	#else
	if ((pEmb[id].EmbType == EMBTYPE_ROOTFS) || (pEmb[id].EmbType == EMBTYPE_USER) || (pEmb[id].EmbType == EMBTYPE_USERRAW)) {
	#endif
		size -= 64;
	}
	#endif
	//check partition size
	if(size > PartitionSize)
	{
		printf("Partition[%d] Size is too smaller than that you wanna update.(0x%08X > 0x%08llX)\r\n"
			,id
			,size
			,PartitionSize);
		return -1;
	}

	ret = nvt_chk_all_in_one_valid(pEmb[id].EmbType, (ulong)p_mem->p_data, p_mem->len, id);
	#endif /* CONFIG_NVT_LINUX_SPINAND_BOOT || CONFIG_NVT_LINUX_SPINORNAND_BOOT || CONFIG_NVT_LINUX_SPINOR_BOOT || CONFIG_NVT_LINUX_EMMC_BOOT*/
	return ret;
}

#ifdef CONFIG_NVT_IVOT_EMMC
static void nvt_switch_boot_patition(unsigned short FlashType, bool switch_back)
{
	char command[128];

	if (FlashType != FLASHTYPE_EMMC)
		return;

	if (switch_back) {
		sprintf(command, "mmc partconf %d 1 %d 0", CONFIG_NVT_IVOT_EMMC, CONFIG_NVT_IVOT_EMMC_BOOT_PART);
		printf("%s, %s\n",__func__,command);
		run_command(command, 0);
		return;
	}

	if (CONFIG_NVT_IVOT_EMMC_BOOT_PART == 7)
		sprintf(command, "mmc partconf %d 1 %d 0", CONFIG_NVT_IVOT_EMMC, CONFIG_NVT_IVOT_EMMC_BOOT_PART);
	else
		sprintf(command, "mmc partconf %d 1 %d %d", CONFIG_NVT_IVOT_EMMC, CONFIG_NVT_IVOT_EMMC_BOOT_PART, CONFIG_NVT_IVOT_EMMC_BOOT_PART);

		printf("%s, %s\n",__func__,command);
	run_command(command, 0);
}
#endif

static int nvt_on_partition_enum_update(unsigned int id, NVTPACK_MEM* p_mem, void* p_user_data)
{
	#if defined(CONFIG_NVT_LINUX_SPINAND_BOOT) || defined(CONFIG_NVT_LINUX_SPINORNAND_BOOT) || defined(CONFIG_NVT_LINUX_SPINOR_BOOT) || defined(CONFIG_NVT_LINUX_EMMC_BOOT)
	EMB_PARTITION* pEmb = (EMB_PARTITION*)p_user_data;
	u64 PartitionOffset = pEmb[id].PartitionOffset;
	u64 PartitionSize = pEmb[id].PartitionSize;

	#ifdef AB_MODIFIED
	#ifdef CONFIG_FW_MULTI_PARTITION_ENABLE
	static unsigned int is_multifw = 0;
	#endif
	#endif

	int ret = 0;

	unsigned int is_secure = 0;
#if (defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT))
	NVT_SMC_EFUSE_DATA efuse_data = {0};
	efuse_data.cmd = NVT_SMC_EFUSE_IS_SECURE;
	is_secure = nvt_ivot_optee_efuse_operation(&efuse_data);
#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
	is_secure = is_secure_enable();
#endif

	switch(pEmb[id].EmbType)
	{
		case EMBTYPE_MBR:
			printf("%s Update: MBR(EMMC only) %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			nvt_chk_last_ebr((ulong)p_mem->p_data, PartitionOffset, pEmb);
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_LOADER:
			printf("%s Update: loader %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
#ifdef CONFIG_NVT_IVOT_EMMC
			nvt_switch_boot_patition(pEmb[id].FlashType, false);
#endif
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
#ifdef CONFIG_NVT_IVOT_EMMC
			nvt_switch_boot_patition(pEmb[id].FlashType, true);
#endif
			break;
		case EMBTYPE_FDT:
			printf("%s Update: fdt %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
#ifdef CONFIG_ENCRYPT_DTS
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT) && defined(CONFIG_ENCRYPT_UBOOT)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				nvt_dbg(IND, "Verify encrypted dts PASS\r\n");
#endif
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_UBOOT:
			printf("%s Update: uboot %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			if(is_secure == 1){
#ifdef CONFIG_ENCRYPT_UBOOT
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT) && defined(CONFIG_ENCRYPT_UBOOT)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				nvt_dbg(IND, "Verify encrypted uboot PASS\r\n");
#endif
			}
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_ATF:
			printf("%s Update: ATF %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			if(is_secure == 1){
#ifdef CONFIG_ENCRYPT_ATF
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT) && defined(CONFIG_ENCRYPT_ATF)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				nvt_dbg(IND, "Verify encrypted ATF PASS\r\n");
#endif
			}
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_APP:
			printf("%s Update: App %u %s \r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize,&pEmb[id]);
			break;
		case EMBTYPE_LINUX:
			printf("%s Update: linux %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			if(is_secure == 1){
#ifdef CONFIG_ENCRYPT_KERNEL
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT) && defined(CONFIG_ENCRYPT_KERNEL)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				nvt_dbg(IND, "Verify encrypted linux PASS\r\n");
#endif
			}
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			#ifdef AB_MODIFIED
			#ifdef CONFIG_FW_MULTI_PARTITION_ENABLE
            if(is_multifw)
			{
			    printf("update linux2\n");
			    ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, pEmb[id+MULTI_FW_PARTITION_OFFSET].PartitionOffset, pEmb[id+MULTI_FW_PARTITION_OFFSET].PartitionSize);
			}
			#endif
			#endif
			break;
		case EMBTYPE_ROOTFS:
        case EMBTYPE_ROOTFS1:
			printf("%s Update: rootfs%u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
#if  defined(NVT_LINUX_RAMDISK_SUPPORT) && defined(CONFIG_ENCRYPT_ROOTFS)
			if((is_secure == 1) && (pEmb[id].OrderIdx == 0)) {
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			} else {
				ret = nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, &pEmb[id]);
			}
#else
			ret = nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, &pEmb[id]);
			#ifdef AB_MODIFIED
			#ifdef CONFIG_FW_MULTI_PARTITION_ENABLE
			if(is_multifw && (pEmb[id].OrderIdx<2))
			{
			    printf("%s Update:another rootfs%u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			    ret = nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, pEmb[id+MULTI_FW_PARTITION_OFFSET].PartitionOffset, pEmb[id+MULTI_FW_PARTITION_OFFSET].PartitionSize,&pEmb[id]);
			}
			#endif
			#endif
#endif
			break;
		case EMBTYPE_RES:
            printf("%s Update: resource %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize,&pEmb[id]);
			break;
		case EMBTYPE_ROOTFSL:
			printf("%s Update: rootfs logical %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, &pEmb[id]);
			break;
		case EMBTYPE_RTOS:
			printf("%s Update: rtos %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			if(is_secure == 1){
#ifdef CONFIG_ENCRYPT_RTOS
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				nvt_dbg(IND, "Verify encrypted rtos PASS\r\n");
#endif
			}
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_TEEOS:
			printf("%s Update: tee os  %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			if(is_secure == 1){
#ifdef CONFIG_ENCRYPT_OPTEE
	#if defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
				if(nvt_chk_signature((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
				if(nvt_chk_signature_on_uboot((ulong)p_mem->p_data, p_mem->len)) {
					printf("check signature fail\n");
					return -1;
				}
	#endif
				nvt_dbg(IND, "Verify encrypted tee os PASS\r\n");
#endif
			}
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_AI:
			printf("%s Update: ai %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_USER:
			printf("%s Update: user partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			nvt_update_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, &pEmb[id]);
			break;
		#ifdef AB_MODIFIED
		#ifdef CONFIG_FW_MULTI_PARTITION_ENABLE
		case EMBTYPE_USER_1:
			is_multifw = 1;
		case EMBTYPE_USER_2:
		case EMBTYPE_USER_3:
			printf("%s Update: AB user partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		#endif
		#endif
		case EMBTYPE_USERRAW:
			printf("%s Update: user raw partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_MCU:
			printf("%s Update: 8051 partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		case EMBTYPE_FDTFAST:
			/* fdt fast is always non secure */
			printf("%s Update: fdtfast %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_update_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize);
			break;
		default:
			printf("%s Update: skip partition %d, embtype=%d %s\r\n", ANSI_COLOR_YELLOW, id, pEmb[id].EmbType, ANSI_COLOR_RESET);
			break;
	}

	#else
		int ret = 0;
		printf("nvt_on_partition_enum_update, not support boot type\n");
	#endif /* CONFIG_NVT_LINUX_SPINAND_BOOT || CONFIG_NVT_LINUX_SPINORNAND_BOOT || CONFIG_NVT_LINUX_SPINOR_BOOT || CONFIG_NVT_LINUX_EMMC_BOOT */
	return ret;
}

static int nvt_on_partition_enum_mtd_readback(unsigned int id, NVTPACK_MEM* p_mem, void* p_user_data)
{
	int ret = 0;
	EMB_PARTITION* pEmb = (EMB_PARTITION*)p_user_data;
	u64 PartitionOffset = pEmb[id].PartitionOffset;
	u64 PartitionSize = pEmb[id].PartitionSize;

	unsigned int is_secure = 0;
#if (defined(CONFIG_NVT_IVOT_OPTEE_SUPPORT) && defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT))
	NVT_SMC_EFUSE_DATA efuse_data = {0};
	efuse_data.cmd = NVT_SMC_EFUSE_IS_SECURE;
	is_secure = nvt_ivot_optee_efuse_operation(&efuse_data);
#elif defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
	is_secure = is_secure_enable();
#endif

	#ifdef AB_MODIFIED
	#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
	ab_fwup_led_toggle();
	#endif
	#endif

	nvt_dbg(IND, "Read back check callback\n");
	switch(pEmb[id].EmbType)
	{
		case EMBTYPE_MBR:
			printf("%s Read: MBR %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_LOADER:
			printf("%s Read: loader %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
#ifdef CONFIG_NVT_IVOT_EMMC
			nvt_switch_boot_patition(pEmb[id].FlashType, false);
#endif
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
#ifdef CONFIG_NVT_IVOT_EMMC
			nvt_switch_boot_patition(pEmb[id].FlashType, true);
#endif
			break;
		case EMBTYPE_FDT:
			printf("%s Read: fdt %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_UBOOT:
			printf("%s Read: uboot %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_ATF:
			printf("%s Read: ATF %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_APP:
			printf("%s Read: App partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_read_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, id, &pEmb[id]);
			break;
		case EMBTYPE_LINUX:
			printf("%s Read: linux %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_ROOTFS:
        case EMBTYPE_ROOTFS1:
        case EMBTYPE_RES:
			printf("%s Read: rootfs%u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
#if  defined(NVT_LINUX_RAMDISK_SUPPORT) && defined(CONFIG_ENCRYPT_ROOTFS)
			if((is_secure == 1) && (pEmb[id].OrderIdx == 0)) {
				ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			} else {
				ret = nvt_read_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, id, &pEmb[id]);
			}
#else
			ret = nvt_read_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, id, &pEmb[id]);
#endif
			break;
		case EMBTYPE_ROOTFSL:
			printf("%s Read: rootfsl%u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_read_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, id, &pEmb[id]);
			break;
		case EMBTYPE_RTOS:
			printf("%s Read: rtos %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_TEEOS:
			printf("%s Read: tee os %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_AI:
			printf("%s Read: ai %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_USER:
			printf("%s Read: user partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_read_fs_partition(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, PartitionSize, id, &pEmb[id]);
			break;
		#ifdef AB_MODIFIED
        case EMBTYPE_USER_1:
        case EMBTYPE_USER_2:
        case EMBTYPE_USER_3:
			printf("%s Read: AB user partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
        #endif
		case EMBTYPE_USERRAW:
			printf("%s Read: user raw partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_MCU:
			printf("%s Read: mcu partition %u %s\r\n", ANSI_COLOR_YELLOW, pEmb[id].OrderIdx, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		case EMBTYPE_FDTFAST:
			printf("%s Read: fdtfast %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
			ret = nvt_read_partitions(pEmb[id].FlashType, (ulong)p_mem->p_data, p_mem->len, PartitionOffset, id, pEmb[id].EmbType);
			break;
		default:
			printf("%s Read: skip partition %d, embtype=%d %s\r\n", ANSI_COLOR_YELLOW, id, pEmb[id].EmbType, ANSI_COLOR_RESET);
			break;
	}

	return ret;
}

static int nvt_getfdt_emb_addr_size(ulong addr, u32 *fdt_emb_addr, u32 *fdt_emb_len)
{
	const char *ptr = NULL;
	const struct fdt_property* fdt_property = NULL;
	int  nodeoffset, nextoffset;

	#if defined(CONFIG_NVT_IVOT_EMMC)
	char path[128] = {0};
	sprintf(path, "mmc%d", CONFIG_NVT_IVOT_EMMC);
	nodeoffset = fdt_path_offset((const void*)addr, path);
	#elif defined(CONFIG_NVT_SPI_NOR) || defined(CONFIG_NVT_SPI_NOR_NAND)
	nodeoffset = fdt_path_offset((const void*)addr, "/nor");
	#else
	nodeoffset = fdt_path_offset((const void*)addr, "/nand");
	#endif
	nextoffset = fdt_first_subnode((const void*)addr, nodeoffset);
	if (nextoffset < 0 || nodeoffset < 0) {
		*fdt_emb_addr = 0;
		return -1;
	}
	*fdt_emb_addr = addr + nextoffset;
	nodeoffset = nextoffset;

	ptr = fdt_get_name((const void*)addr, nextoffset, NULL);
	if( ptr == NULL) {
		printf("%s:%d fdt_get_name got NULL\n", __FILE__, __LINE__);
		return -1;
	}

	while(strcmp(ptr, "nvtpack") != 0) {
		nextoffset = fdt_next_node((const void*)addr, nodeoffset, NULL);
		if (nextoffset < 0) {
			return -1;
		}

		ptr = fdt_get_name((const void*)addr, nextoffset, NULL);
		if( ptr == NULL) {
			printf("%s:%d fdt_get_name got NULL\n", __FILE__, __LINE__);
			return -1;
		}

		nodeoffset = nextoffset;
	}
	nextoffset = fdt_first_property_offset((const void*)addr, nodeoffset);
	nodeoffset = nextoffset;
	if (nextoffset < 0) {
		return -1;
	}
	while(nextoffset >= 0) {
		nextoffset = fdt_next_property_offset((const void*)addr, nodeoffset);
		fdt_property = fdt_get_property_by_offset((const void*)addr, nodeoffset, NULL);
		nodeoffset = nextoffset;
	}

	if (fdt_emb_len != NULL)
		*fdt_emb_len = (ulong)fdt_property->data + strlen(fdt_property->data) - *fdt_emb_addr;

	return 0;
}

static int nvt_checkfdt_part_is_match(const void *fdt_old, const void *fdt_new)
{
	int i, len;
	const void *fdt[2] = {fdt_old, fdt_new};
	const char *name[2] = {0};
	unsigned long long ofs[2] = {0};
	unsigned long long size[2] = {0};
	int  nodeoffset[2] = {0};
#if defined(CONFIG_NVT_IVOT_EMMC)
	char path[128] = {0};
#endif

	for (i = 0; i < 2; i++) {
		#if defined(CONFIG_NVT_IVOT_EMMC)
		sprintf(path, "mmc%d", CONFIG_NVT_IVOT_EMMC);
		nodeoffset[i] = fdt_path_offset((const void*)fdt[i], path);
		#elif defined(CONFIG_NVT_SPI_NOR) || defined(CONFIG_NVT_SPI_NOR_NAND)
		nodeoffset[i] = fdt_path_offset((const void*)fdt[i], "/nor");
		#else
		nodeoffset[i] = fdt_path_offset((const void*)fdt[i], "/nand");
		#endif
		nodeoffset[i] = fdt_first_subnode((const void*)fdt[i], nodeoffset[i]);
		if (nodeoffset[i] < 0) {
			printf(ANSI_COLOR_RED "cannot find mmc or /nor or /nand from fdt.\r\n");
			return -1;
		}
	}

	while (nodeoffset[0] >= 0 || nodeoffset[1] >= 0) {
		//get next nodes of partition_
		for (i = 0; i < 2; i++) {
			ofs[i] = 0;
			size[i] = 0;
			while(nodeoffset[i] >= 0) {
				unsigned long long *nodep;
				name[i] = fdt_get_name(fdt[i], nodeoffset[i], NULL);
				if (strncmp(name[i], "partition_", 10) == 0) {
					nodep = (unsigned long long *)fdt_getprop(fdt[i], nodeoffset[i], "reg", &len);
					if (nodep == NULL || len != sizeof(unsigned long long)*2) {
						printf(ANSI_COLOR_RED "unable to get 'reg' form %s\r\n", name[i]);
						return -1;
					}
					ofs[i] = be64_to_cpu(nodep[0]);
					size[i] = be64_to_cpu(nodep[1]);
					nodeoffset[i] = fdt_next_subnode(fdt[i], nodeoffset[i]);
					break;
				}
				nodeoffset[i] = fdt_next_subnode(fdt[i], nodeoffset[i]);
				name[i] = NULL; //clean name for indicate if partition_xxx has found
			}
		}
		if (name[0] == NULL && name[1] == NULL) {
			break;
		}
		if (strcmp(name[0], name[1]) != 0) {
			printf(ANSI_COLOR_RED "partition name not matched old:%s new:%s\r\n", name[0], name[1]);
			return -1;
		}
		if (ofs[0] != ofs[1]) {
			printf(ANSI_COLOR_RED "%s offset not matched old:%llx new:%llx\r\n", name[0], ofs[0], ofs[1]);
			return -1;
		}
		if (size[0] != size[1]) {
			printf(ANSI_COLOR_RED "%s size not matched old:%llx new:%llx\r\n", name[0], size[0], size[1]);
			return -1;
		}
	}
	return 0;
}

int nvt_process_all_in_one(ulong addr, unsigned int size, int firm_upd_firm)
{
	u32 i;
	int ret = 0;
	unsigned char md5_output_orig[16] = {0};
	unsigned char md5_output_cur[16] = {0};
	u32 need_chk_all_partition_exist = 0;
	NVTPACK_VERIFY_OUTPUT np_verify = {0};
	NVTPACK_GET_PARTITION_INPUT np_get_input;
	NVTPACK_ENUM_PARTITION_INPUT np_enum_input;
	NVTPACK_MEM mem_in = {(void*)addr, (unsigned int)size};
	NVTPACK_MEM mem_out = {0};

	memset(&np_get_input, 0, sizeof(np_get_input));
	memset(&np_enum_input, 0, sizeof(np_enum_input));

	if(nvtpack_verify(&mem_in, &np_verify) != NVTPACK_ER_SUCCESS)
	{
		printf("verify failed.\r\n");
		return -1;
	}
	if(np_verify.ver != NVTPACK_VER_16072017)
	{
		printf("wrong all-in-one bin version\r\n");
		return -1;
	}

	//check if modelext exists.
	EMB_PARTITION* pEmbNew = emb_partition_info_data_new;
	u32 emb_addr_new = 0;
	u32 emb_size_new = 0;
	np_get_input.id = 1; // modelext must always put in partition[1]
	np_get_input.mem = mem_in;
	if(nvtpack_get_partition(&np_get_input,&mem_out) == NVTPACK_ER_SUCCESS)
	{
#if defined(CONFIG_ENCRYPT_DTS)
		/* Prevent source data being overwritten */
		unsigned long tmp_addr = nvt_memory_cfg[MEMTYPE_LINUXTMP].addr;
		memcpy((void*)tmp_addr, (void*)mem_out.p_data, size);
		mem_out.p_data = (void *)tmp_addr;
	#if defined(CONFIG_NVT_IVOT_SECBOOT_SUPPORT)
		if (is_secure_enable() != 0) {
			if(nvt_chk_signature_on_uboot((ulong)mem_out.p_data, mem_out.len))
			{
				nvt_dbg(ERR, "check fdt signature fail\n");
				return -1;

			}
			ret = nvt_decrypt_aes_on_uboot((unsigned long)mem_out.p_data);
			if (ret < 0) {
				nvt_dbg(ERR, "Decrypt fdt fail, ret = %d\r\n", ret);
				return ret;
			}
		}
	#elif defined(CONFIG_NVT_IVOT_OPTEE_SECBOOT_SUPPORT)
		NVT_SMC_EFUSE_DATA efuse_data = {0};
		efuse_data.cmd = NVT_SMC_EFUSE_IS_SECURE;
		if (nvt_ivot_optee_efuse_operation(&efuse_data) != 0) {
			if(nvt_chk_signature((ulong)mem_out.p_data, mem_out.len))
			{
				nvt_dbg(ERR, "check fdt signature fail\n");
				return -1;

			}
			ret = nvt_decrypt_aes((unsigned long)mem_out.p_data);
			if (ret < 0) {
				nvt_dbg(ERR, "Decrypt fdt fail, ret = %d\r\n", ret);
				return ret;
			}
		}
	#endif /* CONFIG_NVT_IVOT_SECBOOT_SUPPORT */
		nvt_dbg(IND, "Verify encrypted dts PASS\r\n");
#endif /* CONFIG_ENCRYPT_DTS */
		if(nvt_check_isfdt((ulong)mem_out.p_data) != 0) {
			printf("partition[1] is not fdt.\r\n");
			return -1;
		}

		/* extract new partition from new modelext */
		ret = nvt_getfdt_emb_addr_size((ulong)mem_out.p_data, &emb_addr_new, &emb_size_new);
		if(ret < 0) {
			printf("failed to get new partition address and size.\r\n");
			return -1;
		}

		ret = nvt_getfdt_emb((ulong)mem_out.p_data, pEmbNew);
		if(ret < 0) {
			printf("failed to get new partition table.\r\n");
			return -1;
		}
	}

	// get current partition in embbed storage
	EMB_PARTITION* pEmbCurr = emb_partition_info_data_curr;
	u32 emb_addr_curr = 0;
	u32 emb_size_curr = 0;
	if (nvt_chk_mtd_fdt_is_null()) {
		printf("%s modelext is empty %s\r\n", ANSI_COLOR_RED, ANSI_COLOR_RESET);
		need_chk_all_partition_exist = 1;
	} else if (nvt_check_isfdt((ulong)nvt_fdt_buffer) == 0) {
		//To do current modelext init.
		ret = nvt_fdt_init(true);
		if (ret < 0)
			printf("fdt init fail\n");

		ret = nvt_getfdt_emb_addr_size((ulong)nvt_fdt_buffer, &emb_addr_curr, &emb_size_curr);
		if(ret < 0) {
			printf("failed to get new partition resource.\r\n");
			return -1;
		}

		ret = nvt_getfdt_emb((ulong)nvt_fdt_buffer, pEmbCurr);
		if(ret < 0) {
			printf("failed to get current partition resource.\r\n");
			return -1;
		}
		// check if partition changes.
		if(emb_size_new != 0 && nvt_checkfdt_part_is_match(nvt_fdt_buffer, mem_out.p_data) != 0) {
			//must be fully all-in-one-bin
			printf("%s detected partition changed. %s\r\n", ANSI_COLOR_RED, ANSI_COLOR_RESET);
			debug("%s fdt in mtd: 0x%08x@0x%08x fdt in all-in-one:0x%08x@0x%08x %s\r\n",
																		ANSI_COLOR_RED,
																		emb_size_curr,
																		emb_addr_curr,
																		emb_size_new,
																		emb_addr_new,
																		ANSI_COLOR_RESET);
			need_chk_all_partition_exist = 1;
		} else {
			printf("%s Partition is not changed %s\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
		}
	} else {
		if (emb_size_new == 0) {
			//also pEmbCurr is NULL
			printf("%s FDT mtd partition doesn't exist in neither embedded nor all-in-one %s\r\n", ANSI_COLOR_RED, ANSI_COLOR_RESET);
			return -1;
		} else {
			//the first burn, must be fully all-in-one-bin
			printf("%s detected partition changed. %s\r\n", ANSI_COLOR_RED, ANSI_COLOR_RESET);
			need_chk_all_partition_exist = 1;
		}
	}
	//read fdt from loader to get fdt info
	//need get some fdt information, like nvt_memory_cfg nsmem in nvt_on_partition_enum_sanity
	// when run the nvt_fdt_init(false); the nvt_fdt_buffer will become new fdt.
	ret = nvt_fdt_init(false);
	if(ret < 0)
	{
		printf("fdt init fail\n");
		return ret;
	}

	//check all-in-one firmware sanity
	EMB_PARTITION* pEmb = (emb_size_new != 0)? pEmbNew : pEmbCurr;
	np_enum_input.mem = mem_in;
	np_enum_input.p_user_data = pEmb;
	np_enum_input.fp_enum = nvt_on_partition_enum_sanity;
	if(nvtpack_enum_partition(&np_enum_input) != NVTPACK_ER_SUCCESS)
	{
		printf("failed sanity.\r\n");
		return -1;
	}

	if(need_chk_all_partition_exist)
	{
		for(i=1; i<EMB_PARTITION_INFO_COUNT; i++)
		{
			if(pEmbNew[i].PartitionSize!=0 && (pEmbNew[i].OrderIdx==0 || pEmbNew[i].EmbType==EMBTYPE_MBR))
			{
				switch(pEmbNew[i].EmbType)
				{
					case EMBTYPE_MBR:
					case EMBTYPE_FDT:
					case EMBTYPE_UBOOT:
					case EMBTYPE_ATF:
					//case EMBTYPE_DSP:
					case EMBTYPE_LINUX:
					case EMBTYPE_ROOTFS:
                    case EMBTYPE_ROOTFS1:
                    case EMBTYPE_RES:
					case EMBTYPE_RTOS:
					case EMBTYPE_APP:
					case EMBTYPE_TEEOS:
					case EMBTYPE_AI:
					case EMBTYPE_MCU:
					case EMBTYPE_FDTFAST:
					{
						//check this type exist in all-in-one
						np_get_input.id = i;
						np_get_input.mem = mem_in;
						if(nvtpack_check_partition(pEmbNew[i], &np_get_input,&mem_out) != NVTPACK_ER_SUCCESS)
						{
							printf("partition changed, need partition[%d]\n",i);
							return -1;
						}
					}break;
				}
			}
		}
	}

	/* partition table env init */
	ret = nvt_part_config(NULL, pEmb);
	if (ret < 0)
		return ret;

	/*
	 * start to update each partition of all-in-one
	 */

	//multi-bin
	np_enum_input.mem = mem_in;
	np_enum_input.p_user_data = pEmb;
	np_enum_input.fp_enum = nvt_on_partition_enum_update;

	if(nvtpack_enum_partition(&np_enum_input) != NVTPACK_ER_SUCCESS)
	{
		printf("failed to run nvt_update_partitions.\r\n");
		return -1;
	}
	/* To calculate original buffer the all-in-one image md5 sum */
	md5_wd((unsigned char *) addr, size, md5_output_orig, CHUNKSZ_MD5);

	/*
	 * Read back updated image from mtd to check if valid or not
	 */
	/*if (firm_upd_firm)
		nvt_disable_mem_protect();*/

	//multi-bin
	np_enum_input.mem = mem_in;
	np_enum_input.p_user_data = pEmb;
	np_enum_input.fp_enum = nvt_on_partition_enum_mtd_readback;
	if(nvtpack_enum_partition(&np_enum_input) != NVTPACK_ER_SUCCESS) {
		printf("failed to run nvt_read_partitions.\r\n");
		return -1;
	}

	//check all-in-one firmware sanity
	np_enum_input.mem = mem_in;
	np_enum_input.p_user_data = pEmb;
	np_enum_input.fp_enum = nvt_on_partition_enum_sanity;

	if(nvtpack_enum_partition(&np_enum_input) != NVTPACK_ER_SUCCESS)
	{
		printf("failed sanity.\r\n");
#ifndef CONFIG_NVT_SPI_NONE
		ret = nvt_flash_mark_bad();
		if (ret)
			printf("flash mark bad process failed\n");
#endif
		return -1;
	} else {
		printf("%s Read back check sanity successfully. %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
	}

	#ifdef AB_MODIFIED
	#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
	ab_fwup_led_toggle();
	#endif
	#endif

	/* To calculate the all-in-one image md5 sum after read back */
	md5_wd((unsigned char *) addr, size, md5_output_cur, CHUNKSZ_MD5);

	#ifdef AB_MODIFIED
	#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
	ab_fwup_led_toggle();
	#endif
	#endif

	if (memcmp(md5_output_orig, md5_output_cur, sizeof(md5_output_cur)) != 0) {
		printf("%s All-in-one image MD5 sum is not match %s\n", ANSI_COLOR_RED, ANSI_COLOR_RESET);
#ifndef CONFIG_NVT_SPI_NONE
		ret = nvt_flash_mark_bad();
		if (ret)
			printf("flash mark bad process failed\n");
#endif
		return -1;
	} else {
		printf("%s All-in-one image MD5 sum is match %s\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
	}

	if(emb_size_new!=0)
	{
		int ret = 0;
		//reload new modelext but must keep loader information
		printf("Reload new modelext partition after updating\r\n");
		ret = nvt_fdt_init(true);
		if (ret < 0)
			printf("modelext init fail\n");
	}
	return 0;
}

/*
 * This function can be used to do system recovery in emmc boot mode
 * Below condition will do system recovery:
 *				Loader send event
 */
int nvt_process_sys_recovery(void)
{
	int ret = 0;

#if defined(CONFIG_NVT_IVOT_EMMC_T)
	char dev_part_str[10];
	loff_t size = 0;
	u32 i;
	ulong addr = 0;
	MODELEXT_HEADER* p_resource_curr = NULL;
	EMB_PARTITION* pEmbCurr = NULL;
	NVTPACK_GET_PARTITION_INPUT np_get_input;

	memset(&np_get_input, 0, sizeof(np_get_input));

	printf("%sStarting to do EMMC boot recovery %s\r\n", ANSI_COLOR_YELLOW, ANSI_COLOR_RESET);
	/* Switch to emmc bus partition 3 (cache partition) */

	sprintf(dev_part_str, "%d:3", CONFIG_NVT_IVOT_EMMC);
	if (fs_set_blk_dev("mmc", dev_part_str, FS_TYPE_FAT)) {
		return ERR_NVT_UPDATE_OPENFAILED;
	} else {
		if (!fs_exists(get_nvt_bin_name(NVT_BIN_NAME_TYPE_RECOVERY_FW))) {
			return ERR_NVT_UPDATE_NO_NEED;
		}
	}
	if (fs_set_blk_dev("mmc", dev_part_str, FS_TYPE_FAT))
		return ERR_NVT_UPDATE_OPENFAILED;
	else {
		ret = fs_read(get_nvt_bin_name(NVT_BIN_NAME_TYPE_RECOVERY_FW), (ulong)nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, 0, 0, &size);
		if (size <= 0 || ret < 0) {
			printf("Read %s at 0x%x failed ret=%d\n", get_nvt_bin_name(NVT_BIN_NAME_TYPE_RECOVERY_FW), nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, size);
			return ERR_NVT_UPDATE_READ_FAILED;
		} else {
			printf("Read %s at 0x%x successfully, size=%d\n", get_nvt_bin_name(NVT_BIN_NAME_TYPE_RECOVERY_FW), nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, size);
			NVTPACK_MEM mem_in = {(void*)nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, (unsigned int)size};
			NVTPACK_MEM mem_out = {0};
			pEmbCurr = (EMB_PARTITION*)modelext_get_cfg((unsigned char*)nvt_memory_cfg[MEMTYPE_SHMEM].addr,MODELEXT_TYPE_EMB_PARTITION,&p_resource_curr);
			if(pEmbCurr==NULL) {
				printf("failed to get current partition resource.\r\n");
				return -1;
			}
			for(i=1; i<EMB_PARTITION_INFO_COUNT; i++)
			{
				if(pEmbCurr[i].PartitionSize!=0 && (pEmbCurr[i].OrderIdx==0 || pEmbCurr[i].EmbType==EMBTYPE_MBR || pEmbCurr[i].EmbType==EMBTYPE_DSP))
				{
					switch(pEmbCurr[i].EmbType)
					{
						case EMBTYPE_MBR:
						case EMBTYPE_FDT:
						case EMBTYPE_ATF:
						case EMBTYPE_UBOOT:
						#if defined(CONFIG_DSP1_FREERTOS) || defined(CONFIG_DSP2_FREERTOS)
						case EMBTYPE_DSP:
						#endif
						case EMBTYPE_LINUX:
						case EMBTYPE_ROOTFS:
						case EMBTYPE_RTOS:
						case EMBTYPE_TEEOS:
						case EMBTYPE_AI:
						case EMBTYPE_MCU:
						case EMBTYPE_FDTFAST:
						{
							//check this type exist in all-in-one
							np_get_input.id = i;
							np_get_input.mem = mem_in;
							if(nvtpack_check_partition(pEmbCurr[i], &np_get_input,&mem_out) != NVTPACK_ER_SUCCESS)
							{
								printf("Recovery procedure: need partition[%d]\n",i);
								return -1;
							}
						}break;
					}
				}
			}
			ret = nvt_process_all_in_one((ulong)nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr ,size, 0);
			if (ret < 0)
				return ERR_NVT_UPDATE_FAILED;
		}
	}
#endif

	return ret;
}

int nvt_fw_update(bool firm_upd_firm)
{
	int ret = 0, retry = 0;
	SHMINFO *p_shminfo;
	loff_t size = 0;
	char command[128];

	p_shminfo = (SHMINFO *)nvt_memory_cfg[MEMTYPE_SHMEM].addr;
	if (strncmp(p_shminfo->boot.LdInfo_1, "LD_NVT", 6) != 0) {
		printf("can't get right bin info in %s \n", __func__);
		return ERR_NVT_UPDATE_FAILED;
	}

#ifdef CONFIG_NVT_IVOT_SOC_BOOT_FROM_RAM
	unsigned long firm_addr = nvt_readl((ulong)nvt_shminfo_comm_fw_update_addr);
	unsigned long firm_size = nvt_readl((ulong)nvt_shminfo_comm_fw_update_len);
	ulong fdt_addr = nvt_readl((ulong)nvt_shminfo_boot_fdt_addr);
	ulong fdt_size = fdt_totalsize(fdt_addr);

	if (size > nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size) {
		printf("%sYour image size %lld is larger than all-in-one size with %d %s\r\n", ANSI_COLOR_RED, size, nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size, ANSI_COLOR_RESET);
		return ERR_NVT_UPDATE_READ_FAILED;
	}

	/* To check if all-in-one buffer is overlapped with mem-tbl's fdt */
	if ((fdt_addr + fdt_size) > nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr && fdt_addr < (nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr + nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size)) {

		printf("%s nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr (0x%lX) is overlapped with fdt(0x%lX-0x%lX), check nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size%s\r\n",
				ANSI_COLOR_RED, nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, fdt_addr, fdt_addr + fdt_size, ANSI_COLOR_RESET);
		return ERR_NVT_UPDATE_READ_FAILED;
	}

	/**
	 * To check if all-in-one buffer is overlapped with mem-tbl's shmem
	 * DO NOT use static-check (such as #if), otherwise it may cause compiler error for non-all-in-one vendor.
	 */
	if (nvt_memory_cfg[MEMTYPE_SHMEM].addr + nvt_memory_cfg[MEMTYPE_SHMEM].size > nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr) {
		printf("%s nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr (0x%lX) is overlapped with shmem(0x%lX-0x%lX), check nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size%s\r\n",
				ANSI_COLOR_RED, nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, nvt_memory_cfg[MEMTYPE_SHMEM].addr, nvt_memory_cfg[MEMTYPE_SHMEM].addr + nvt_memory_cfg[MEMTYPE_SHMEM].size, ANSI_COLOR_RESET);
		return ERR_NVT_UPDATE_READ_FAILED;
	}

	memmove((void *)nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, (void *)firm_addr, firm_size);
#else
	
	if (firm_upd_firm) {
		#ifdef AB_MODIFIED
		#ifdef CONFIG_FW_MULTI_PARTITION_ENABLE
		sprintf(command, "abpwm led_fwup_start");
		run_command(command, 0);
		#endif
		#endif
		unsigned long firm_addr = nvt_readl((ulong)nvt_shminfo_comm_fw_update_addr);
		unsigned long firm_size = nvt_readl((ulong)nvt_shminfo_comm_fw_update_len);

		#ifdef AB_MODIFIED
		#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
		ab_fwup_led_toggle_init();
		sprintf(command, "abpwm led_fwup_proc");
		run_command(command, 0);
		#endif
		#endif
		printf("%s\tfirmware image at: 0x%08lx@0x%08lx %s\r\n", ANSI_COLOR_YELLOW, firm_size, firm_addr, ANSI_COLOR_RESET);
		ret = nvt_process_all_in_one(firm_addr, firm_size, firm_upd_firm);
		if (ret < 0)
			return ERR_NVT_UPDATE_FAILED;
	} else if (p_shminfo->boot.LdCtrl2 & LDCF_BOOT_FLASH) {
		return ERR_NVT_UPDATE_NO_NEED;
	} else {
		#ifdef AB_MODIFIED
		#ifdef CONFIG_FW_MULTI_PARTITION_ENABLE
		sprintf(command, "abpwm led_fwup_start");
		run_command(command, 0);
		#endif
		#endif
		if (nvt_fs_set_blk_dev())
			return ERR_NVT_UPDATE_OPENFAILED;
		else {
			if (fs_size(get_nvt_bin_name(NVT_BIN_NAME_TYPE_FW), &size) < 0) {
				return ERR_NVT_UPDATE_NO_NEED;
			}
		}
		
		while (retry < UPDATE_ALL_IN_ONE_RETRY_TIMES) {
		if (nvt_fs_set_blk_dev())
			return ERR_NVT_UPDATE_OPENFAILED;
		else {
			ulong fdt_addr = nvt_readl((ulong)nvt_shminfo_boot_fdt_addr);
			ulong fdt_size = fdt_totalsize(fdt_addr);
			/* To check if A.bin is larger than all-in-one temp buffer size */
			if (size > nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size) {
				printf("%sYour image size %lld is larger than all-in-one size with %lu %s\r\n", ANSI_COLOR_RED, size, nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size, ANSI_COLOR_RESET);
				return ERR_NVT_UPDATE_READ_FAILED;
			}
			/* To check if all-in-one buffer is overlapped with mem-tbl's fdt */
			if (fdt_addr + fdt_size > nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr) {
				printf("%s nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr (0x%lX) is overlapped with fdt(0x%lX-0x%lX), check nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size%s\r\n",
					ANSI_COLOR_RED, nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, fdt_addr, fdt_addr + fdt_size, ANSI_COLOR_RESET);
				return ERR_NVT_UPDATE_READ_FAILED;
			}
			/**
			 * To check if all-in-one buffer is overlapped with mem-tbl's shmem
			 * DO NOT use static-check (such as #if), otherwise it may cause compiler error for non-all-in-one vendor.
			 */
			if (nvt_memory_cfg[MEMTYPE_SHMEM].addr + nvt_memory_cfg[MEMTYPE_SHMEM].size > nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr) {
				printf("%s nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr (0x%lX) is overlapped with shmem(0x%lX-0x%lX), check nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].size%s\r\n",
					ANSI_COLOR_RED, nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, nvt_memory_cfg[MEMTYPE_SHMEM].addr, nvt_memory_cfg[MEMTYPE_SHMEM].addr + nvt_memory_cfg[MEMTYPE_SHMEM].size, ANSI_COLOR_RESET);
				return ERR_NVT_UPDATE_READ_FAILED;
			}

			ret = fs_read(get_nvt_bin_name(NVT_BIN_NAME_TYPE_FW), (ulong)nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, 0, 0, &size);
			if (size <= 0 || ret < 0) {
				printf("Read %s at 0x%lx failed ret=%lld\n", get_nvt_bin_name(NVT_BIN_NAME_TYPE_FW), nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, size);
				return ERR_NVT_UPDATE_READ_FAILED;
			} else {
				#ifdef AB_MODIFIED
				#ifdef CONFIG_AB_UPGRADE_LED_SINGLE
				ab_fwup_led_toggle_init();
				sprintf(command, "abpwm led_fwup_proc");
				run_command(command, 0);
				#endif
				#endif
				printf("Read %s at 0x%lx successfully, size=%lld\n", get_nvt_bin_name(NVT_BIN_NAME_TYPE_FW), nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr, size);
				ret = nvt_process_all_in_one((ulong)nvt_memory_cfg[MEMTYPE_ALL_IN_ONE].addr ,size, 0);
				if (ret < 0) {
					retry++;
				} else {
					/* This flag is for the first boot after update A.bin finished. */
					p_shminfo->boot.LdCtrl2 &= ~LDCF_BOOT_CARD;
					break;
				}
			}
		}
		if (retry >= UPDATE_ALL_IN_ONE_RETRY_TIMES)
			return ERR_NVT_UPDATE_FAILED;
		}
	}
#endif
	return 0;
}
