/*
 * Novatek PWM driver on NVT_FT platform.
 *
 * Copyright (C) 2020 Novatek MicroElectronics Corp.
 *
 * SPDX-License-Identifier: GPL-2.0+
 */

// #include <common.h>
#include <div64.h>
#include <pwm.h>
#include <asm/io.h>
#include <linux/libfdt.h>
#include "nvt_abpwm.h"
#include <command.h>
#include <config.h>
#include <asm/nvt-common/nvt_common.h>

/* PWM Control Register */
#define PWM_CTRL_REG(id)     ((0x08 * id) + 0x00)
#define PWM_CYCLE_MASK       GENMASK(15, 0)
#define PWM_TYPE_BIT         BIT(16)

/* PWM Period Register */
#define PWM_PRD_REG(id)      ((0x08 * id) + 0x04)
#define PWM_RISE_MASK        GENMASK(7, 0)
#define PWM_FALL_MASK        GENMASK(15, 8)
#define PWM_PERIOD_MASK      GENMASK(23, 16)
#define PWM_INV_BIT          BIT(28)

/* PWM Expand Period Register */
#define PWM_PRD_HI_REG(id)   ((0x04 * id) + 0x230)
#define PWM_RISE_HI_MASK     GENMASK(7, 0)
#define PWM_FALL_HI_MASK     GENMASK(15, 8)
#define PWM_PERIOD_HI_MASK   GENMASK(23, 16)

/* PWM Enable Register */
#define ABPWM_EN_REG           0x100
/* PWM Disable Register */
#define ABPWM_DIS_REG          0x104

#define SRC_FREQ             120000000
#define PCLK_FREQ            250000

#define	PIN_PWM_CFG_PWM0_1     0x1         ///< PWM0_1  (P_GPIO[0])
#define	PIN_PWM_CFG_PWM0_2     0x2         ///< PWM0_2  (DSI_GPIO[6])
#define	PIN_PWM_CFG_PWM1_1     0x10        ///< PWM1_1  (P_GPIO[1])
#define	PIN_PWM_CFG_PWM1_2     0x20        ///< PWM1_2  (DSI_GPIO[7])
#define	PIN_PWM_CFG_PWM2_1     0x100       ///< PWM2_1  (P_GPIO[2])
#define	PIN_PWM_CFG_PWM2_2     0x200       ///< PWM2_2  (DSI_GPIO[8])
#define	PIN_PWM_CFG_PWM3_1     0x1000      ///< PWM3_1  (P_GPIO[3])
#define	PIN_PWM_CFG_PWM3_2     0x2000      ///< PWM3_2  (DSI_GPIO[9])
#define	PIN_PWM_CFG_PWM4_1     0x10000     ///< PWM4_1  (P_GPIO[4])
#define	PIN_PWM_CFG_PWM4_2     0x20000     ///< PWM4_2  (D_GPIO[2])
#define	PIN_PWM_CFG_PWM5_1     0x100000    ///< PWM5_1  (P_GPIO[5])
#define	PIN_PWM_CFG_PWM5_2     0x200000    ///< PWM5_2  (D_GPIO[3])

#define PIN_PWMII_CFG_PWM6_1   0x1         ///< PWM6_1  (P_GPIO[6])
#define PIN_PWMII_CFG_PWM6_2   0x2         ///< PWM6_2  (D_GPIO[4])
#define PIN_PWMII_CFG_PWM7_1   0x10        ///< PWM7_1  (P_GPIO[7])
#define PIN_PWMII_CFG_PWM7_2   0x20        ///< PWM7_2  (D_GPIO[5])
#define PIN_PWMII_CFG_PWM8_1   0x100       ///< PWM8_1  (P_GPIO[8])
#define PIN_PWMII_CFG_PWM8_2   0x200       ///< PWM8_2  (C_GPIO[4])
#define PIN_PWMII_CFG_PWM9_1   0x1000      ///< PWM9_1  (P_GPIO[9])
#define PIN_PWMII_CFG_PWM9_2   0x2000      ///< PWM9_2  (C_GPIO[5])
#define PIN_PWMII_CFG_PWM10_1  0x10000     ///< PWM10_1 (P_GPIO[10])
#define PIN_PWMII_CFG_PWM10_2  0x20000     ///< PWM10_2 (C_GPIO[6])
#define PIN_PWMII_CFG_PWM11_1  0x100000    ///< PWM11_1 (P_GPIO[11])
#define PIN_PWMII_CFG_PWM11_2  0x200000    ///< PWM11_2 (C_GPIO[7])

u32 abpclk_freq = PCLK_FREQ;
u32 pwm_pinmux = 0;

static inline void nvt_abpwm_write_reg(int reg, u32 val)
{
	writel(val, IOADDR_PWM_REG_BASE + reg);
}

static inline u32 abnvt_pwm_read_reg(int reg)
{
	return readl(IOADDR_PWM_REG_BASE + reg);
}

static void ab_fwup_led_pwm_info(uint32_t* pwm_id,uint32_t* pwm_id2,uint32_t* active)
{
	uint32_t *cell = NULL;
	ulong fdt_addr = nvt_readl((ulong)nvt_shminfo_boot_fdt_addr);
	int nodeoffset = fdt_path_offset((const void*)fdt_addr, "/ab_uboot_fwup_led");
	if (nodeoffset >= 0) {

		cell = (uint32_t*)fdt_getprop((const void*)fdt_addr, nodeoffset, "fwup_led_pwmid", NULL);
		if (cell > 0) {
			*pwm_id= be32_to_cpu(cell[0]);
			//printf("Get fwup_led_pwmid = %d\n",*pwm_id);
		}
		else
		{
			printf("Get fwup_led_pwmid fail\n");
		}
		
		cell = (uint32_t*)fdt_getprop((const void*)fdt_addr, nodeoffset, "fwup_led_pwmid2", NULL);
		if (cell > 0) {
			*pwm_id2= be32_to_cpu(cell[0]);
			//printf("Get fwup_led_pwmid2 = %d\n",*pwm_id2);
		}
		else
		{
			printf("Get fwup_led_pwmid2 fail\n");
		}
		
		cell = (uint32_t*)fdt_getprop((const void*)fdt_addr, nodeoffset, "led_pwm_active", NULL);
		if (cell > 0) {
			*active= be32_to_cpu(cell[0]);
			//printf("Get led_pwm_active = %d\n",*active);
		}
		else
		{
			printf("Get led_pwm_active fail\n");
		}
		
	}
	else
	{
		ab_err_print("get nodeoffset of ab_uboot_fwup_led fail.\n");
	}
}

int do_nvt_abpwm_set_pinmux(int pwm_id,int pwm_id2)
{
	char pwm[10] = {0};
    //u32 pwm_pinmux = 0;


	if (pwm_id < 6) {
		sprintf(pwm, "pwm");
		if(pwm_id2>0 && pwm_id2<3)
		{
			switch(pwm_id)
			{
				case 0:
					pwm_pinmux = PIN_PWM_CFG_PWM0_1*pwm_id2;
					break;
				case 1:
					pwm_pinmux = PIN_PWM_CFG_PWM1_1*pwm_id2;
					break;
				case 2:
					pwm_pinmux = PIN_PWM_CFG_PWM2_1*pwm_id2;
					break;
				case 3:
					pwm_pinmux = PIN_PWM_CFG_PWM3_1*pwm_id2;
					break;
				case 4:
					pwm_pinmux = PIN_PWM_CFG_PWM4_1*pwm_id2;
					break;
				case 5:
					pwm_pinmux = PIN_PWM_CFG_PWM5_1*pwm_id2;
					break;
			}
		}
		else
		{
			pwm_pinmux = 0x0;
		}
	} else {
		sprintf(pwm, "pwmII");
		if(pwm_id2>0 && pwm_id2<3)
		{
			switch(pwm_id)
			{
				case 6:
					pwm_pinmux = PIN_PWMII_CFG_PWM6_1*pwm_id2;
					break;
				case 7:
					pwm_pinmux = PIN_PWMII_CFG_PWM7_1*pwm_id2;
					break;
				case 8:
					pwm_pinmux = PIN_PWMII_CFG_PWM8_1*pwm_id2;
					break;
				case 9:
					pwm_pinmux = PIN_PWMII_CFG_PWM9_1*pwm_id2;
					break;
				case 10:
					pwm_pinmux = PIN_PWMII_CFG_PWM10_1*pwm_id2;
					break;
				case 11:
					pwm_pinmux = PIN_PWMII_CFG_PWM11_1*pwm_id2;
					break;
			}
		}
		else
		{
			pwm_pinmux = 0x0;
		}
	}
	printf("%s_pinmux = 0x%x\n", pwm, pwm_pinmux);
	nvt_pwm_set_pinmux(pwm_id, pwm_pinmux);

	return 0;
}

void nvt_abpwm_reset_pinmux(int pwm_id, u32 pwm_pinmux)
{
	if (pwm_id == 0) {
		if (pwm_pinmux & PIN_PWM_CFG_PWM0_1) {
			//DBG_ERR("Before:\nValue[0]=0x%x\n,Value[1]=0x%x\n,Value[2]=0x%x\n", *(u32 *)(IOADDR_TOP_REG_BASE + 0x18), *(u32 *)(IOADDR_TOP_REG_BASE + 0x18),*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWM_CFG_PWM0_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xD8) &= (0x1 << 6);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
            DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
		//DBG_ERR("After:\nValue[0]=0x%x\n,Value[1]=0x%x\n,Value[2]=0x%x\n", *(u32 *)(IOADDR_TOP_REG_BASE + 0x18), *(u32 *)(IOADDR_TOP_REG_BASE + 0x18),*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8));
	} else if (pwm_id == 1) {
		if (pwm_pinmux & PIN_PWM_CFG_PWM1_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWM_CFG_PWM1_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xD8) &= (0x1 << 7);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 2) {
		if (pwm_pinmux & PIN_PWM_CFG_PWM2_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWM_CFG_PWM2_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xD8) &= (0x1 << 8);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 3) {
		if (pwm_pinmux & PIN_PWM_CFG_PWM3_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWM_CFG_PWM3_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xD8) &= (0x1 << 9);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 4) {
		if (pwm_pinmux & PIN_PWM_CFG_PWM4_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWM_CFG_PWM4_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xB4) &= (0x1 << 2);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 5) {
		if (pwm_pinmux & PIN_PWM_CFG_PWM5_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWM_CFG_PWM5_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xB4) &= (0x1 << 3);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 6) {
		if (pwm_pinmux & PIN_PWMII_CFG_PWM6_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWMII_CFG_PWM6_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xB4) &= (0x1 << 4);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 7) {
		if (pwm_pinmux & PIN_PWMII_CFG_PWM7_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x1 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWMII_CFG_PWM7_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) |= (0x2 << (pwm_id * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xB4) &= (0x1 << 5);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << (pwm_id * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 8) {
		if (pwm_pinmux & PIN_PWMII_CFG_PWM8_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x1 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWMII_CFG_PWM8_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x2 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA0) &= (0x1 << 4);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << ((pwm_id - 8) * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 9) {
		if (pwm_pinmux & PIN_PWMII_CFG_PWM9_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x1 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWMII_CFG_PWM9_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x2 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA0) &= (0x1 << 5);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << ((pwm_id - 8) * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 10) {
		if (pwm_pinmux & PIN_PWMII_CFG_PWM10_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x1 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWMII_CFG_PWM10_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x2 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA0) &= (0x1 << 6);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << ((pwm_id - 8) * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	} else if (pwm_id == 11) {
		if (pwm_pinmux & PIN_PWMII_CFG_PWM11_1) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x1 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA8) &= (0x1 << pwm_id);
		} else if (pwm_pinmux & PIN_PWMII_CFG_PWM11_2) {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) &= (0x7 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x1C) |= (0x2 << ((pwm_id - 8) * 4));
			*(u32 *)(IOADDR_TOP_REG_BASE + 0xA0) &= (0x1 << 7);
		} else {
			*(u32 *)(IOADDR_TOP_REG_BASE + 0x18) &= (0x7 << ((pwm_id - 8) * 4));
			DBG_ERR("!!Caution!!>>using pwm_id(%d) but pwm_pinmux(0x%x) not configured(check nvt-top.dtsi)\n", pwm_id, pwm_pinmux);
		}
	}
}

int abpwm_init(int pwm_id,int pwm_id2, int div, int invert)
{
	u32 prd_reg = 0x0;

	/* Enable clk */
	pwm_enable_clk(pwm_id);

	/* Enable pinmux depends on dts */
	
	// this need create a new function
	do_nvt_abpwm_set_pinmux(pwm_id,pwm_id2);

	/* Disable pwm before inverting */
	nvt_abpwm_write_reg(ABPWM_DIS_REG, (u32)(0x1 << pwm_id));

	/* Invert pwm */
	prd_reg = abnvt_pwm_read_reg(PWM_PRD_REG(pwm_id));

	if (invert == 1) {
		prd_reg |= PWM_INV_BIT;
	} else {
		prd_reg &= ~PWM_INV_BIT;
	}

	nvt_abpwm_write_reg(PWM_PRD_REG(pwm_id), prd_reg);

	return 0;
}

void nvt_abpwm_set_freq(int pwm_id, u32 freq)
{
	u32 div;

	if (!freq || (freq > SRC_FREQ)) {
		DBG_ERR("pwm_id(%d) Invalid pwm frequency\r\n", pwm_id);
		return;
	}

	div = SRC_FREQ / freq;
	div = div - 1;
	abpclk_freq = freq;

	printf("abpclk_freq(%u) div(%u)\n", freq, div);

	nvt_pwm_set_div(pwm_id, div);
}

int abpwm_config(int pwm_id, int duty_ns, int period_ns)
{
	u32 prd_reg = 0x0;
	u32 prd_hi_reg = 0x0;
	u64 period = 0, duty = 0;

	/* Set free run mode */
	nvt_abpwm_write_reg(PWM_CTRL_REG(pwm_id), (u32)0x0);

	period = (u64)abpclk_freq * (u64)period_ns;
	do_div(period, 1000000000);

	duty = (u64)abpclk_freq * (u64)duty_ns;
	do_div(duty, 1000000000);

	prd_reg = abnvt_pwm_read_reg(PWM_PRD_REG(pwm_id));

	prd_reg &= ~PWM_RISE_MASK;  /* Output is high since started */

	prd_reg &= ~PWM_FALL_MASK;
	prd_reg |= (PWM_FALL_MASK & (duty << 8));

	prd_reg &= ~PWM_PERIOD_MASK;
	prd_reg |= (PWM_PERIOD_MASK & (period << 16));

	nvt_abpwm_write_reg(PWM_PRD_REG(pwm_id), prd_reg);

	if (pwm_id < 8) {
		prd_hi_reg &= ~PWM_RISE_HI_MASK;  /* Output is high since started */

		prd_hi_reg &= ~PWM_FALL_HI_MASK;
		prd_hi_reg |= (PWM_FALL_HI_MASK & ((duty >> 8) << 8));

		prd_hi_reg &= ~PWM_PERIOD_HI_MASK;
		prd_hi_reg |= (PWM_PERIOD_HI_MASK & ((period >> 8) << 16));

		nvt_abpwm_write_reg(PWM_PRD_HI_REG(pwm_id), prd_hi_reg);
	}

	return 0;
}

int abpwm_enable(int pwm_id)
{
	nvt_abpwm_write_reg(ABPWM_EN_REG, (u32)(0x1 << pwm_id));

	//TODO: wait_for_completion
	return 0;
}

void abpwm_disable(int pwm_id)
{
	nvt_abpwm_write_reg(ABPWM_DIS_REG, (u32)(0x1 << pwm_id));
	
	nvt_abpwm_reset_pinmux(pwm_id,pwm_pinmux);
	//TODO: wait_for_completion
}

static int do_abpwm(struct cmd_tbl *cmdtp, int flag, int argc, char *const argv[])
{
	static int led_fwup_step = 0; //0:disable, 1:start , 2:processing
	
	uint32_t pwmid=0,pwmid2=0,active=0;
	ab_fwup_led_pwm_info(&pwmid,&pwmid2,&active);
	if(!active)
	{
		printf("UBOOT LED PWM is not active.\n");
		return 0;
	}
	
	if (!strncmp(argv[1], "led_fwup_start", 14)) 
	{
		if(argc>2 && argc<5)
		{
			int pwm_id  = atoi(argv[2]);
			int pwm_id2 = atoi(argv[3]);
			if(led_fwup_step==0)
			{
				nvt_abpwm_set_freq(pwm_id, 10000); /* 10K */
				abpwm_init(pwm_id,pwm_id2, 0, 0);
				abpwm_config(pwm_id, 200000000/*get_duty_ns(0)*/, 200000000/*get_period_ns(0)*/);
				abpwm_enable(pwm_id);
				led_fwup_step =1;
			}
		}
		else
		{
			if(led_fwup_step==0)
			{
				nvt_abpwm_set_freq(pwmid, 10000); /* 10K */
				abpwm_init(pwmid,pwmid2, 0, 0);
				abpwm_config(pwmid, 200000000/*get_duty_ns(0)*/, 200000000/*get_period_ns(0)*/);
				abpwm_enable(pwmid);
				led_fwup_step =1;
			}
		}
		return 0;
	}
	else if (!strncmp(argv[1], "led_fwup_proc", 13)) 
	{
		if(argc>2 && argc<5)
		{
			int pwm_id  = atoi(argv[2]);
			int pwm_id2 = atoi(argv[3]);
			if(led_fwup_step==1)
			{
				abpwm_config(pwm_id, 100000000/*get_duty_ns(0)*/, 200000000/*get_period_ns(0)*/);
				abpwm_enable(pwm_id);
				led_fwup_step =2;
			}
		}
		else
		{
			if(led_fwup_step==1)
			{
				abpwm_config(pwmid, 100000000/*get_duty_ns(0)*/, 200000000/*get_period_ns(0)*/);
				abpwm_enable(pwmid);
				led_fwup_step =2;
			}
		}
		return 0;
	}
	else if (!strncmp(argv[1], "disable", 7)) 
	{
		if(argc>2 && argc<5)
		{
			int pwm_id  = atoi(argv[2]);
			if(led_fwup_step!=0)
			{
				abpwm_disable(pwm_id);
				led_fwup_step =0;
			}
		}
		else
		{
			if(led_fwup_step!=0)
			{
				abpwm_disable(pwmid);
				led_fwup_step =0;
			}
		}
		return 0;
	}
	
	return CMD_RET_USAGE;
}

U_BOOT_CMD(
	abpwm,  6,  0,   do_abpwm,
	"abpwm operation",
	"abpwm led_fwup_start\n"
	"abpwm led_fwup_start [pwd_id] [pwd_id2]\n"
	"abpwm led_fwup_proc\n"
	"abpwm led_fwup_proc [pwd_id] [pwd_id2]\n"
	"abpwm disable\n"
	"abpwm disable [pwd_id]\n"
	"Note:\n"
	"Default freq is 10KHz,\n"
);
