/*
 *  driver/power/reset/na51102-poweroff.c
 *  Novatek     Power Button Control driver
 *
 *  Author:     robin_hsu@novatek.com.tw
 *  Created:	April 13, 2022
 *  Copyright:	Novatek Inc.
 *
 */

#include <linux/module.h>
#include <linux/rtc.h>
#include <linux/platform_device.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/soc/nvt/nvt-io.h>
#include <plat/rtc_reg.h>
#include <plat/pwbc_reg.h>
#include <plat/pwbc_int.h>
#include <plat/rtc_int.h>
#include <plat/hardware.h>
#include <linux/clk.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/semaphore.h>
#include <linux/proc_fs.h>
#include <linux/of.h>
#include <linux/reboot.h>
#include <linux/pm.h>

#define DRV_VERSION		"1.01.002"

#ifdef CONFIG_RTC_NVT_PWBC_SDT_USER
#define RTC_PWBC_SDT_ISR_DISABLE	1
#else
#define RTC_PWBC_SDT_ISR_DISABLE	0
#endif

//static struct completion pwbc_cset_completion;
static struct semaphore pwbc_sem;
static void __iomem *_REGIOBASE_PWBC;

#define loc_cpu() down(&pwbc_sem);
#define unl_cpu() up(&pwbc_sem);

#ifndef AB_MODIFIED
#define AB_MODIFIED
#endif

#ifdef AB_MODIFIED
#define AB_PWBC_COMPATIBLE_NAME "ab,ab_pwbc_sw1"
uint32_t u_ab_nvt_disable_sw1_forced_shutdown = 0;

#define AB_PWBC_PWR_OFF_CTRL_ENABLE 1
#endif
void nvt_pwbc_wait_cset_done(void);
void nvt_pwbc_trigger_cset(void);
void nvt_pwbc_reset_shutdown_timer(void);
void nvt_pwbc_power_off(void);

void __iomem* hyload_base ;

struct nvt_pwbc_priv {
	struct device *pwbc;
	struct proc_dir_entry *pproc_pwbc;
	#ifdef AB_MODIFIED
    struct proc_dir_entry *pproc_psrc;
    #endif
    struct delayed_work cset_work;
};

static void pwbc_setreg(uint32_t offset, REGVALUE value)
{
	nvt_writel(value, _REGIOBASE_PWBC + offset);
}

static REGVALUE pwbc_getreg(uint32_t offset)
{
	return nvt_readl(_REGIOBASE_PWBC + offset);
}

void nvt_pwbc_trigger_cset(void)
{
    ulong timeout;
    union PWBC_CTRL2_REG pwbcctrl2_reg;

	/*Wait for PWBC is ready for next CSET*/
    timeout = jiffies + msecs_to_jiffies(1000);
	do {
		pwbcctrl2_reg.reg = pwbc_getreg(PWBC_CTRL2_REG_OFS);
        if (time_after_eq(jiffies, timeout)) {
			printk("PWBC CSET timeout, plz check RTC 32K OSC, PWR_SW or VCC_VBAT\n");
			return;
		}
	} while (pwbcctrl2_reg.bit.pwbc_cset == 1);

	/*Trigger CSET*/
	pwbcctrl2_reg.reg = pwbc_getreg(PWBC_CTRL2_REG_OFS);
	pwbcctrl2_reg.bit.pwbc_cset = 1;
	pwbc_setreg(PWBC_CTRL2_REG_OFS, pwbcctrl2_reg.reg);
}

void nvt_pwbc_wait_cset_done(void)
{
	union PWBC_STS_REG pwbc_sts_reg;
    ulong timeout;
    
    timeout = jiffies + msecs_to_jiffies(1000);
    do
	{
		pwbc_sts_reg.reg = pwbc_getreg(PWBC_STS_REG_OFS);
        if (time_after_eq(jiffies, timeout)) {
			printk("PWBC CSET timeout, plz check RTC 32K OSC, PWR_SW or VCC_VBAT\n");
			return;
		}
	} while(pwbc_sts_reg.bit.pwbc_cset_done == 0);

    // clear cset done
    pwbc_sts_reg.bit.pwbc_cset_done =1;
    pwbc_setreg(PWBC_STS_REG_OFS, pwbc_sts_reg.reg);
}

void nvt_pwbc_reset_shutdown_timer(void)
{
    ulong timeout;
    union PWBC_CTRL_REG pwbcctrl_reg;
    union PWBC_CTRL2_REG pwbcctrl2_reg;

	/*Wait for previous CSET done */
    timeout = jiffies + msecs_to_jiffies(1000);
	do {
		pwbcctrl2_reg.reg = pwbc_getreg(PWBC_CTRL2_REG_OFS);
        if (time_after_eq(jiffies, timeout)) {
			printk("PWBC CSET timeout, plz check RTC 32K OSC, PWR_SW or VCC_VBAT\n");
			return;
		}
	} while (pwbcctrl2_reg.bit.pwbc_cset == 1);

	pwbcctrl2_reg.reg = pwbc_getreg(PWBC_CTRL2_REG_OFS);
	pwbcctrl2_reg.bit.pwbc_cset = 0;
	pwbcctrl2_reg.bit.seq_time1_sel = 1;
	pwbc_setreg(PWBC_CTRL2_REG_OFS, pwbcctrl2_reg.reg);

	pwbcctrl_reg.reg = pwbc_getreg(PWBC_CTRL_REG_OFS);
	pwbcctrl_reg.bit.reset_sdt_timer = 1;
	pwbc_setreg(PWBC_CTRL_REG_OFS, pwbcctrl_reg.reg);

    nvt_pwbc_trigger_cset();
    nvt_pwbc_wait_cset_done();
}

void nvt_pwbc_power_off(void)
{
    ulong timeout;
    union PWBC_CTRL_REG pwbcctrl_reg;
    union PWBC_CTRL2_REG pwbcctrl2_reg;

	/*Wait for previous CSET done */
    timeout = jiffies + msecs_to_jiffies(1000);
	do {
		pwbcctrl2_reg.reg = pwbc_getreg(PWBC_CTRL2_REG_OFS);
        if (time_after_eq(jiffies, timeout)) {
			printk("PWBC CSET timeout, plz check RTC 32K OSC, PWR_SW or VCC_VBAT\n");
			return;
		}
	} while (pwbcctrl2_reg.bit.pwbc_cset == 1);
    
	pwbcctrl2_reg.reg = pwbc_getreg(PWBC_CTRL2_REG_OFS);
	pwbcctrl2_reg.bit.pwbc_cset = 0;
	pwbcctrl2_reg.bit.seq_time1_sel = 1;
	pwbc_setreg(PWBC_CTRL2_REG_OFS, pwbcctrl2_reg.reg);

	pwbcctrl_reg.reg = pwbc_getreg(PWBC_CTRL_REG_OFS);
	pwbcctrl_reg.bit.pwr_off = 1;
	pwbc_setreg(PWBC_CTRL_REG_OFS, pwbcctrl_reg.reg);

    nvt_pwbc_trigger_cset();
    nvt_pwbc_wait_cset_done();
}


static irqreturn_t nvt_pwbc_update_handler(int irq, void *data)
{
    union PWBC_CTRL_REG pwbcctrl_reg;
    union PWBC_STS_REG pwbcsts_reg;
    
	pwbcsts_reg.reg = pwbc_getreg(PWBC_STS_REG_OFS);
	pwbcctrl_reg.reg = pwbc_getreg(PWBC_CTRL_REG_OFS);

    pwbcsts_reg.reg &= pwbcctrl_reg.reg;

	if (pwbcsts_reg.reg) {
		pwbc_setreg(PWBC_STS_REG_OFS, pwbcsts_reg.reg);

		if (pwbcsts_reg.bit.pwr_sw1_int_sts) {
			// reset shutdown timer to prevent SW1 shutdown system
			if (pwbcctrl_reg.bit.reset_sdt_timer == 0) {
            	nvt_pwbc_reset_shutdown_timer();
			}
		}
	}
	return IRQ_HANDLED;
}

static int nvt_pwbc_init_sw(void)
{
#ifdef AB_MODIFIED
    if(u_ab_nvt_disable_sw1_forced_shutdown == 1) {
       	//trigger ISR to disable shutdown timer.
		union PWBC_CTRL_REG pwbc_ctrl_reg;
		printk("[%s %d] disable sw1 forced shutdown.\n", __func__, __LINE__);

		pwbc_ctrl_reg.reg = pwbc_getreg(PWBC_CTRL_REG_OFS);
		pwbc_ctrl_reg.bit.pwr_sw1_inten = 1;
		pwbc_setreg(PWBC_CTRL_REG_OFS, pwbc_ctrl_reg.reg);
    } else {
		printk("[%s %d] keep sw1 forced shutdown.\n", __func__, __LINE__);
    }
	return 0;
#else
#if (RTC_PWBC_SDT_ISR_DISABLE == 0)
	union PWBC_CTRL_REG pwbc_ctrl_reg;

	pwbc_ctrl_reg.reg = pwbc_getreg(PWBC_CTRL_REG_OFS);
	pwbc_ctrl_reg.bit.pwr_sw1_inten = 1;
	pwbc_setreg(PWBC_CTRL_REG_OFS, pwbc_ctrl_reg.reg);
#endif
	return 0;

#endif
}


static void nvt_pwbc_add_one_sec(u32* sec, u32* min, u32* hour, u32* days)
{
	if (*sec == 59) {
		if (*min == 59) {
			*hour += 1;
			*min = 0;
			*sec = 0;
		} else {
			*min += 1;
			*sec = 0;
		}
	} else {
		*sec += 1;
	}

	if (*hour > 23) {
		*hour = 0;
		*days += 1;
	}
}

void nvt_pwbc_power_control(int reboot_sec)
{
	struct rtc_time cur_time;
    struct rtc_wkalrm alrm;
    u32 sec, min, hour, days;
    int i;

    if (reboot_sec) { // set rtc power alarm

        nvt_rtc_read_time(NULL, &cur_time);
        alrm.time = cur_time;

        sec  = cur_time.tm_sec;
        min  = cur_time.tm_min;
    	hour = cur_time.tm_hour;
    	days = cur_time.tm_mday;

        for (i = 0;i < reboot_sec; i++)
    		nvt_pwbc_add_one_sec(&sec, &min, &hour, &days);

        alrm.time.tm_sec  = sec;
        alrm.time.tm_min  = min;
        alrm.time.tm_hour = hour;
        alrm.time.tm_mday = days;
        alrm.enabled      = 1;

         // set pwr_alarm time
         nvt_rtc_set_alarm(NULL, &alrm);
	}
    loc_cpu();
    
    // set reset shutdown timer and power off
    nvt_pwbc_reset_shutdown_timer();
    nvt_pwbc_power_off();
    unl_cpu();
    
    printk("%s: If not power off, plz check 32K crystal, pad PWR_EN or VCC_VBAT\r\n", __func__);
}

static int nvt_pwbc_proc_show(struct seq_file *seq, void *v)
{
	union PWBC_STS_REG pwbcsts_reg;

	pwbcsts_reg.reg = pwbc_getreg(PWBC_STS_REG_OFS);
	seq_printf(seq, "power switch 1 value %d\n", pwbcsts_reg.bit.pwr_sw1_pin);
	seq_printf(seq, "power switch 2 value %d\n", pwbcsts_reg.bit.pwr_sw2_pin);
	seq_printf(seq, "power switch 3 value %d\n", pwbcsts_reg.bit.pwr_sw3_pin);
	seq_printf(seq, "power switch 4 value %d\n", pwbcsts_reg.bit.pwr_sw4_pin);
	seq_printf(seq, "power on source ");
	if (pwbcsts_reg.bit.pwronsrc_sw1) {
		if(pwbcsts_reg.bit.pwronsrc_pwralarm)
			seq_printf(seq, "RTC Alarm\r\n");
		else
			seq_printf(seq, "SW1\r\n");
	} else if (pwbcsts_reg.bit.pwronsrc_sw2) {
		if(pwbcsts_reg.bit.pwronsrc_pwralarm)
			seq_printf(seq, "RTC Alarm\r\n");
		else
			seq_printf(seq, "SW2\r\n");
	} else if (pwbcsts_reg.bit.pwronsrc_sw3) {
		if(pwbcsts_reg.bit.pwronsrc_pwralarm)
			seq_printf(seq, "RTC Alarm\r\n");
		else
			seq_printf(seq, "SW3\r\n");
	} else if (pwbcsts_reg.bit.pwronsrc_sw4) {
		if(pwbcsts_reg.bit.pwronsrc_pwralarm)
			seq_printf(seq, "RTC Alarm\r\n");
		else
			seq_printf(seq, "SW4\r\n");
	} else {
		if(pwbcsts_reg.bit.pwronsrc_pwralarm)
			seq_printf(seq, "RTC Alarm\r\n");
		else
		seq_printf(seq, "None\r\n");
	}
	return 0;
}

#ifdef AB_MODIFIED
static int psrc_proc_show(struct seq_file *seq, void *v)
{
       union PWBC_STS_REG pwbcsts_reg;

       pwbcsts_reg.reg = pwbc_getreg(PWBC_STS_REG_OFS);

       if (pwbcsts_reg.bit.pwronsrc_sw1) {
		   if(pwbcsts_reg.bit.pwronsrc_pwralarm)
				seq_printf(seq, "9\r\n");
			else
               seq_printf(seq, "1\r\n");
       } else if (pwbcsts_reg.bit.pwronsrc_sw2) {
		   if(pwbcsts_reg.bit.pwronsrc_pwralarm)
				seq_printf(seq, "9\r\n");
		   else
               seq_printf(seq, "2\r\n");
       } else if (pwbcsts_reg.bit.pwronsrc_sw3) {
		   if(pwbcsts_reg.bit.pwronsrc_pwralarm)
				seq_printf(seq, "9\r\n");
			else
               seq_printf(seq, "3\r\n");
       } else if (pwbcsts_reg.bit.pwronsrc_sw4) {
		   if(pwbcsts_reg.bit.pwronsrc_pwralarm)
				seq_printf(seq, "9\r\n");
			else
               seq_printf(seq, "4\r\n");
       } else {
			if(pwbcsts_reg.bit.pwronsrc_pwralarm)
				seq_printf(seq, "9\r\n");
			else
               seq_printf(seq, "None\r\n");
       }
       return 0;
}
#endif

static int nvt_pwbc_proc_open(struct inode *inode, struct file *file)
{
    return single_open(file, nvt_pwbc_proc_show, NULL);
}

#ifdef AB_MODIFIED
static int psrc_proc_open(struct inode *inode, struct file *file)
{
        return single_open(file, psrc_proc_show, NULL);
}
#endif

#define PWBC_MAX_CMD_LENGTH 30
static ssize_t nvt_pwbc_proc_cmd_write(struct file *file, const char __user *buf,
		size_t size, loff_t *off)
{
	int val;
	int len = size;
	char cmd_line[PWBC_MAX_CMD_LENGTH];
	char *cmdstr = cmd_line;
	const char delimiters[] = {' ', 0x0A, 0x0D, '\0'};
	char *p_arg;
    union PWBC_CTRL_REG pwbcctrl_reg;

	// check command length
	if (len > (PWBC_MAX_CMD_LENGTH - 1)) {
		printk("%s: Command length %d is too long\n", __func__, len);
		return -EINVAL;
	}

	// copy command string from user space
	if (copy_from_user(cmd_line, buf, len)) {
		return -EINVAL;
	}

	if (len == 0)
		cmd_line[0] = '\0';
	else
		cmd_line[len - 1] = '\0';

	p_arg = strsep(&cmdstr, delimiters);
	sscanf(p_arg, "%x", &val);

    loc_cpu();

	pwbcctrl_reg.reg = pwbc_getreg(PWBC_CTRL_REG_OFS);
    //reset shutdown timer to prevent SW1 shutdown system
	if (pwbcctrl_reg.bit.reset_sdt_timer == 0) {
       nvt_pwbc_reset_shutdown_timer();
	}
    unl_cpu();

	return size;
}

static struct proc_ops proc_pwbc_fops = {
	.proc_open      = nvt_pwbc_proc_open,
	.proc_read      = seq_read,
	.proc_lseek     = seq_lseek,
	.proc_release   = single_release,
	.proc_write     = nvt_pwbc_proc_cmd_write,
};

#ifdef AB_MODIFIED
static struct proc_ops proc_psrc_fops = {
        .proc_open    = psrc_proc_open,
        .proc_read    = seq_read,
        .proc_lseek  = seq_lseek,
        .proc_release = single_release,
        .proc_write   = NULL,
};
#endif

// poweroff
static int nvt_pwbc_restart(struct notifier_block *this, unsigned long action,
		      void *cmd)
{
    nvt_pwbc_power_control(action);
	return NOTIFY_DONE;
}

static struct notifier_block pwbc_restart_nb = {
	.notifier_call = nvt_pwbc_restart,
	.priority = 128,
};

static void nvt_do_pwbc_poweroff(void)
{
	#ifdef AB_MODIFIED
	//disable set rtc alarm
	nvt_pwbc_restart(&pwbc_restart_nb, 0, NULL);
	#else
	nvt_pwbc_restart(&pwbc_restart_nb, 10, NULL);
    #endif
}
#ifdef AB_MODIFIED
static void ab_nvt_pwbc_read_property(void) {

  struct device_node *ps_ab_node = NULL;
  uint32_t u_read_val = 0;
  int i_rst = 0;
  ps_ab_node = of_find_compatible_node(NULL, NULL, AB_PWBC_COMPATIBLE_NAME);
  if(ps_ab_node == NULL) {
      printk("[%s %d] no compatible for %s\n",__FUNCTION__,__LINE__,AB_PWBC_COMPATIBLE_NAME);
      return;
  }
  i_rst = of_property_read_u32(ps_ab_node, "config_disable_linux_forced_shutdown",&u_read_val);
  if (i_rst == 0) {
      u_ab_nvt_disable_sw1_forced_shutdown = u_read_val;
  }

}
#if AB_PWBC_PWR_OFF_CTRL_ENABLE
ssize_t pwbc_ctrl_show(struct device_driver *driver, char *buf) {
    return 0;
}

ssize_t pwbc_ctrl_store(struct device_driver *driver, const char *buf, size_t count) {
    pr_emerg("ab pwbc force power off\n");
    nvt_do_pwbc_poweroff();
    while(1){
        mdelay(5000);
        pr_emerg("ab pwbc not power off yet, try again\n");
        nvt_do_pwbc_poweroff();
    }
    return 1;
}

static DRIVER_ATTR_RW(pwbc_ctrl);

void ab_pwbc_create_pwbc_ctrl_sys_file(void);
void ab_pwbc_remove_pwbc_ctrl_sys_file(void);
#endif
#endif
static int nvt_pwbc_probe(struct platform_device *pdev)
{
    struct nvt_pwbc_priv *priv;
	struct resource *memres = NULL;
	struct proc_dir_entry *pentry = NULL;
    int ret = 0, irq = 0;

    memres = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (unlikely(!memres)) {
		dev_err(&pdev->dev, "failed to get resource\n");
		return -ENXIO;
	}

    _REGIOBASE_PWBC = devm_ioremap_resource(&pdev->dev, memres);;
	if (unlikely(_REGIOBASE_PWBC == 0)) {
		dev_err(&pdev->dev, "failed to get io memory\n");
		goto out;
	}

	priv = kzalloc(sizeof(struct nvt_pwbc_priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

#ifdef AB_MODIFIED
    ab_nvt_pwbc_read_property();
#endif
	platform_set_drvdata(pdev, priv);

	device_init_wakeup(&pdev->dev, 1);

	/*Set default HW configuration*/
	/*Define what data type will be used, RCW_DEF or manual define*/

	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_err(&pdev->dev, "No IRQ resource!\n");
		goto out;
	}

	ret = devm_request_irq(&pdev->dev, irq, nvt_pwbc_update_handler, 0, pdev->name,
			priv);
	if (ret < 0) {
		dev_err(&pdev->dev, "Failed to request IRQ: #%d: %d\n", irq, ret);
		goto out;
	}

	sema_init(&pwbc_sem, 1);

    nvt_pwbc_init_sw();

	pentry = proc_create("pwbc", S_IRUGO | S_IXUGO, NULL, &proc_pwbc_fops);
        if (pentry == NULL) {
                dev_err(&pdev->dev, "failed to create pwbc proc!\n");
                ret = -EINVAL;
                goto out_proc_pwbc;
        }
	priv->pproc_pwbc = pentry;
	
	#ifdef AB_MODIFIED
    pentry = proc_create("psrc", S_IRUGO | S_IXUGO, NULL, &proc_psrc_fops);
        if (pentry == NULL) {
                dev_err(&pdev->dev, "failed to create psrc proc!\n");
                ret = -EINVAL;
                goto out;
        }
       priv->pproc_psrc = pentry;
    #endif
	
    register_restart_handler(&pwbc_restart_nb);

	pm_power_off = nvt_do_pwbc_poweroff;

#ifdef AB_MODIFIED
#if AB_PWBC_PWR_OFF_CTRL_ENABLE
    ab_pwbc_create_pwbc_ctrl_sys_file();
#endif
#endif

	return 0;


out_proc_pwbc:
	if (priv->pproc_pwbc) {
		proc_remove(priv->pproc_pwbc);
	}
out:
    kfree(priv);
	return ret;
}

static int nvt_pwbc_remove(struct platform_device *pdev)
{
	struct nvt_pwbc_priv *priv = platform_get_drvdata(pdev);

	if (priv->pproc_pwbc) {
		proc_remove(priv->pproc_pwbc);
	}

	kfree(priv);

#ifdef AB_MODIFIED
#if AB_PWBC_PWR_OFF_CTRL_ENABLE
    ab_pwbc_remove_pwbc_ctrl_sys_file();
#endif
#endif

	return 0;
}
#ifdef CONFIG_OF
static const struct of_device_id nvt_pwbc_of_dt_ids[] = {
    { .compatible = "nvt,nvt_pwbc", },
    {},
};
MODULE_DEVICE_TABLE(of, nvt_pwbc_of_dt_ids);
#endif

static struct platform_driver nvt_pwbc_platform_driver = {
	.driver		= {
		.name	= "nvt_pwbc",
		.owner	= THIS_MODULE,
		.of_match_table = nvt_pwbc_of_dt_ids,
	},
	.probe		= nvt_pwbc_probe,
	.remove		= nvt_pwbc_remove,
};

static int __init nvt_pwbc_init(void)
{
	int ret;

	ret = platform_driver_register(&nvt_pwbc_platform_driver);
	return ret;
}

static void __exit nvt_pwbc_exit(void)
{
	platform_driver_unregister(&nvt_pwbc_platform_driver);
}

#ifdef AB_MODIFIED
#if AB_PWBC_PWR_OFF_CTRL_ENABLE
void ab_pwbc_create_pwbc_ctrl_sys_file(void) {
    driver_create_file(&nvt_pwbc_platform_driver.driver, &driver_attr_pwbc_ctrl);
}

void ab_pwbc_remove_pwbc_ctrl_sys_file(void) {
    driver_remove_file(&nvt_pwbc_platform_driver.driver, &driver_attr_pwbc_ctrl);
}
#endif
#endif

MODULE_AUTHOR("Novatek");
MODULE_DESCRIPTION("nvt PWBC driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);
MODULE_ALIAS("platform:pwbc-nvt");

module_init(nvt_pwbc_init);
module_exit(nvt_pwbc_exit);

