// SPDX-License-Identifier: GPL-2.0
/*
 * NVT UART
 *
 * Driver for NVT Soc
 *
 * @file  nvt_serial.c
 *
 * Copyright Novatek Microelectronics Corp. 2020. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#if defined(CONFIG_SERIAL_NVT_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/atomic.h>
#include <linux/hrtimer.h>
#include <linux/module.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/irq.h>
#include <linux/init.h>
#include <linux/console.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial_core.h>
#include <linux/serial.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/dma-mapping.h>
#include <linux/clk-provider.h>
#include <linux/soc/nvt/nvt-pcie-lib.h>

#ifdef AB_MODIFIED
#include <linux/spinlock.h>
#include <linux/timekeeping.h>
#include <linux/rtc.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/string.h>
#include <linux/kfifo.h>
#include <linux/kthread.h>
#include <linux/version.h>
#endif

#include <plat/nvt_serial.h>
#include <plat/top.h>

#include<linux/timekeeping.h>
#include<linux/timer.h>

#define DRIVER_VERSION  "3.2.16"

#define DRIVER_NAME     "nvt_serial"

// #define FOR_GPS

static unsigned int nr_uarts = CONFIG_SERIAL_NVT_NR_UARTS;
module_param(nr_uarts, uint, 0444);
MODULE_PARM_DESC(nr_uarts, "Maximum number of UARTs supported. (1-" __MODULE_STRING(CONFIG_SERIAL_NVT_NR_UARTS) ")");

static int global_line = 0;

#if defined(CONFIG_SERIAL_NVT_DMA_RX_SIZE)
static int __serial_rx_size = CONFIG_SERIAL_NVT_DMA_RX_SIZE;
#else
static int __serial_rx_size = -1;
#endif
module_param_named(serial_rx_size, __serial_rx_size, int, 0600);

static int __serial_tx_dma_en_debug = -1;
module_param_named(serial_tx_dma_en_debug, __serial_tx_dma_en_debug, int, 0600);

static int __serial_rx_dma_en_debug = -1;
module_param_named(serial_rx_dma_en_debug, __serial_rx_dma_en_debug, int, 0600);

static int __serial_debug = 0;
module_param_named(serial_debug, __serial_debug, int, 0600);

static int __serial_debug_console = 0;
module_param_named(serial_debug_console, __serial_debug_console, int, 0600);

static int __serial_flow_debug = 0;
module_param_named(serial_flow_debug, __serial_flow_debug, int, 0600);

#define serial_dbg(fmt, ...) do { \
	if (port->line != 0 || __serial_debug_console) { \
		if (__serial_debug) \
			pr_info("nvt_serial%d: "fmt, port->line, ##__VA_ARGS__); \
		else \
			pr_debug("nvt_serial%d: "fmt, port->line,##__VA_ARGS__); \
	} \
} while (0)

#define serial_err(fmt, ...) do { \
	pr_err("nvt_serial%d: "fmt, port->line, ##__VA_ARGS__); \
} while (0)

#define serial_flow_dbg(fmt, ...) do { \
	if (port->line != 0 || __serial_debug_console) { \
		if (__serial_flow_debug) \
			pr_info("nvt_serial%d: %s\n"fmt, port->line, __func__, ##__VA_ARGS__); \
	} \
} while (0)

#ifdef AB_MODIFIED
static DEFINE_SPINLOCK(ab_logdump_lock);
#define AB_LOGDUMP_EN_KFIFO                     1
#define AB_LOGDUMP_EN_POLL_FEATURE              0   /* Unsupport poll feature. This have terrible issue. */
#define AB_LOGDUMP_EN_TIME_MEASUREMENT          0
#define AB_LOGDUMP_EN_BLOCK_READ_TIMEOUT        1

#define AB_LOGDUMP_DEV_NAME                     "logdump"
#define AB_LOGDUMP_DEV_NUM                      1
#define AB_LOGDUMP_DEV_MAJOR_NUM                0
#define AB_LOGDUMP_DEV_MINOR_NUM                0

#define AB_LOGDUMP_BUFF_DATA_MAXSIZE            4084
#define AB_LOGDUMP_BUFF_REMAIN_LIMIT_SIZE       64
#define AB_LOGDUMP_BUFF_STRUCT_SIZE             4096
#define AB_LOGDUMP_INITLOG_MAX_LEN              512

#define AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM       2
#define AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM    14
#define AB_LOGDUMP_DYNAMIC_POOL_TOTAL_SIZE      (AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM*AB_LOGDUMP_BUFF_STRUCT_SIZE)
#define AB_LOGDUMP_BUFF_MAX_CNT                 (AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM+AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM)
#define AB_LOGDUMP_TIMESTAMP_PRE_LEN            5
#define AB_LOGDUMP_TIMESTAMP_POST_MS_LEN        3
#define AB_LOGDUMP_TIMESTAMP_POST_US_LEN        6

#define AB_LOGDUMP_WAKEUP_FREQ_CNT              16
#define AB_LOGDUMP_WAKEUP_TIMEOUT_MS            50
#define AB_LOGDUMP_WAKEUP_TIMEOUT_JIFFIES       (HZ/(1000/AB_LOGDUMP_WAKEUP_TIMEOUT_MS))

#if (LINUX_VERSION_CODE > KERNEL_VERSION(5, 10, 0))
#define _get_boottime(p)	ktime_get_boottime_ts64(p)
#else
#define _get_boottime(p)	get_monotonic_boottime64(p)
#endif

typedef enum {
    AB_LOGDUMP_POOL_MAIN    = 0,
    AB_LOGDUMP_POOL_DYNAMIC = 1,
} AB_LOGDUMP_POOL_ET;

typedef enum {
	AB_LOGDUMP_BUFF_VALID,                           // mean doesn't filled any data
	AB_LOGDUMP_BUFF_IN_USE,                          // mean the buffer still filling data
	AB_LOGDUMP_BUFF_FULL,                            // mean the buffer is full, wait to transfer to usersapce
	AB_LOGDUMP_BUFF_INVALID,                         // mean the buffer is empty, but disable to fill data
} AB_LOGDUMP_BUFF_STATUS_ET;

typedef enum {
    AB_LODDUMP_TIMESTAMP_ACCURACY_MS    = 0,
    AB_LODDUMP_TIMESTAMP_ACCURACY_US    = 1,
} AB_LOGDUMP_TIMESTAMP_ACCURACY_ET;

typedef enum {
    AB_LOGDUMP_SWITCH_BUFF_MECH_SEQUENCE        = 0,
    AB_LOGDUMP_SWITCH_BUFF_MECH_REDIRECT_TOP    = 1,
    AB_LOGDUMP_SWITCH_BUFF_MECH_OVERWRITE_SEQ   = 2,
    AB_LOGDUMP_SWITCH_BUFF_MECH_DROP_SEQ        = 3,
} AB_LOGDUMP_SWITCH_BUFF_MECH_ET;

typedef enum {
    AB_LOGDUMP_FILL_INFOLOG_INIT_INFO       = 0,
    AB_LOGDUMP_FILL_INFOLOG_DROP_INFO       = 1,
    AB_LOGDUMP_FILL_INFOLOG_TIMESTAMP_INFO  = 2,
} AB_LOGDUMP_FILL_INFOLOG_ET;

typedef struct {
    void * ul_next_bufptr;                        // record next buffer address
    uint16_t uw_datalen;                            // record filled data size
    uint8_t  uc_bufstatus;                          // record buffer status, refer AB_LOGDUMP_BUFF_STATUS_ET
    uint8_t  uc_inqueue;                            // record is buffer in queue to transfer or not
    uint8_t  uc_data[AB_LOGDUMP_BUFF_DATA_MAXSIZE]; // record buffer index
} AB_LOGDUMP_BUFF_ST;

typedef struct {
    uint8_t  uc_buff_max_idx;       // range 0 ~ (AB_LOGDUMP_BUFF_MAX_CNT-1 or AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM-1)
    bool b_is_occur_linebreak;      // ture: previous character is line break
    bool b_logbuff_en_overwrite;    // true: when all buffers are full, it should overwrite the oldest buffer. false: drop the data.
    uint64_t ul_drop_log_len;       // the length of dropping log.
    uint32_t ui_overwrite_buff_cnt; // the count of overwrite buffer.
} AB_LOGDUMP_BUFF_INFO_ST;

#if !AB_LOGDUMP_EN_KFIFO
typedef struct {
    int8_t c_input_idx;     // point to newest input element
    int8_t c_output_idx;    // point to newest output element
} AB_LOGDUMP_TRANSFER_QUEUE_INFO_ST;
#endif

typedef struct {
    /* only serial in use */
    bool b_logdump_fillheader;  // true: already fill header, false: not yet

    /* both ioctl,open,close and serial in use */
    bool b_logdump_buff_ready;
    bool b_uartinit_ready;          // true: host do log dump init action, false: not yet

    /* only ioctl,open,close use */
    bool b_logdump_dev_ready;
    bool b_dev_region_ready;
    bool b_cdev_ready;
    
    bool b_logdump_terminate_trig;  // true: terminate to release epoll wait .
    bool b_logdump_forcedump_trig;  // true: need to set current buffer as full and switch to next buffer
    bool b_logdump_en_debuglog;     // true: enable debug log for tracing issue
    AB_LOGDUMP_TIMESTAMP_ACCURACY_ET e_timstamp_accuracy;
    dev_t ui_logdump_dev;
    struct cdev ps_cdev;
    struct device *ps_device;
    struct class *ps_class;
    struct timespec64 s_ts;
#if AB_LOGDUMP_EN_TIME_MEASUREMENT
    struct timespec64 s_measurement[4];
    int32_t ia_measurement[4];
#endif
} AB_LOGDUMP_DESC_INFO_ST;

typedef struct {
	struct spinlock     s_read_lock;
    wait_queue_head_t t_read_wait;
    struct semaphore s_protect_sem;
    bool b_wait_wakeup;             // the flag of blocking read() and enable input part to do wake_up action.
} AB_LOGDUMP_OPEN_INFO_ST;

/* ioctl data structure */
typedef enum {
    AB_LOGDUMP_CTRL_FEATURE_UART_INIT       = 0,    // notify driver host ready
    AB_LOGDUMP_CTRL_FEATURE_FORCE_DUMP_LOG  = 1,    // notify driver to dump remain log
    AB_LOGDUMP_CTRL_FEATURE_TERMINATE       = 2,    // notify driver host need to stop access character device
    AB_LOGDUMP_CTRL_FEATURE_TIMESTAMP_UNIT  = 3,    // set timestamp unit, data value: 0-ms, 1-us (refer AB_LOGDUMP_TIMESTAMP_ACCURACY_ET)
    AB_LOGDUMP_CTRL_FEATURE_EN_DEBUG_LOG    = 4,    // enable debug flow log
    AB_LOGDUMP_CTRL_FEATURE_DUMP_MEASURE_OUTPUT = 5,
} AB_LOGDUMP_CTRL_FEATURE_ET;

#pragma pack(1)
typedef struct {
    uint8_t uc_feature_idx;     // reference AB_LOGDUMP_CTRL_FEATURE_ET
    uint8_t uc_feature_data;    // 
} AB_LOGDUMP_CTRL_DATA_ST;

static void* pv_main_buff_pool = NULL;              // point to main buffer pool
static void* pv_dynamic_buff_pool = NULL;           // point to dynamic buffer pool

static AB_LOGDUMP_BUFF_ST* ps_logbuff = NULL;       // point to buffer 0 of main buffer pool
static AB_LOGDUMP_BUFF_ST* ps_curlogbuff = NULL;    // point to current inused buffer
static AB_LOGDUMP_DESC_INFO_ST* ps_desc = NULL;     // point to log dump descriptor

static int i_devs = AB_LOGDUMP_DEV_NUM;
static unsigned int i_major = AB_LOGDUMP_DEV_MAJOR_NUM;
static unsigned int i_minor = AB_LOGDUMP_DEV_MINOR_NUM;

#if AB_LOGDUMP_EN_KFIFO
static struct kfifo s_transfer_queue = {0};
#else
static AB_LOGDUMP_BUFF_ST* ps_transfer_queue[AB_LOGDUMP_BUFF_MAX_CNT] = {0};
static AB_LOGDUMP_TRANSFER_QUEUE_INFO_ST s_transfer_info = {0};
#endif
static AB_LOGDUMP_BUFF_INFO_ST s_buff_info = {0};
static AB_LOGDUMP_OPEN_INFO_ST *ps_open_info = NULL;

#define IOC_MAGIC       'k'
#define AB_FEATURE_CTRL _IOW(IOC_MAGIC, 1, AB_LOGDUMP_CTRL_DATA_ST*)

#define IOC_MAX_NUMBER  3

#define init_MUTEX(sem)  sema_init(sem, 1)
#define init_SEMA(sem, initval) sema_init(sem, initval)

//SN18210-529(52x): if we do printk(KERN_DEBUG ...) in nvt_tx_handle(IRQ context), 
// the UART may crash ,causing the system to hang.
// to prevent this situation, disable all printk logs from ab_logdump.
//Additional information: please do not printk(directly show log on terminal) in IRQ context, linux system will crash.
//#define AB_LOGDUMP_PRINT(en, _format, args...) if(en) {printk(KERN_DEBUG ""_format, ##args);}
#define AB_LOGDUMP_PRINT(en, _format, args...) {}
//20250626 : it is also useful with printk_deferred, but the (SN18210)customer's request is to keep the above modification.
//#define AB_LOGDUMP_PRINT(en, _format, args...) if(en) {printk_deferred(KERN_DEBUG ""_format, ##args);}

#endif

static void rx_dma_timeout(unsigned long data);

#define UART_TO_NVT(uart_port)	((struct nvt_port *) uart_port)

static void nvt_handle_tx(struct uart_port *port);
static unsigned int nvt_tx_empty(struct uart_port *port);

static unsigned int nvt_read(struct uart_port *port, unsigned int off)
{
	return readl_relaxed(port->membase + off);
}

static void nvt_write(struct uart_port *port, unsigned int off, unsigned int val)
{
	struct nvt_port *nvt_port = container_of(port, struct nvt_port, uart);
	unsigned long flags = 0;

	spin_lock_irqsave(&nvt_port->write_lock, flags);

	writel_relaxed(val, port->membase + off);

	spin_unlock_irqrestore(&nvt_port->write_lock, flags);
}

static void nvt_masked_write(struct uart_port *port, unsigned int off, unsigned int mask, unsigned int val)
{
	struct nvt_port *nvt_port = container_of(port, struct nvt_port, uart);
	unsigned long flags = 0;
	unsigned int tmp = nvt_read(port, off);  /* be careful that some registers are cleared after reading, such as RBR */

	spin_lock_irqsave(&nvt_port->write_lock, flags);

	tmp &= ~mask;
	tmp |= val & mask;
	writel_relaxed(tmp, port->membase + off);

	spin_unlock_irqrestore(&nvt_port->write_lock, flags);
}

static DEFINE_SPINLOCK(write_lock_early);
static void nvt_write_early(struct uart_port *port, unsigned int off, unsigned int val)
{
	unsigned long flags = 0;

	spin_lock_irqsave(&write_lock_early, flags);

	writel_relaxed(val, port->membase + off);

	spin_unlock_irqrestore(&write_lock_early, flags);
}

static void nvt_clear_fifos(struct uart_port *port)
{
	struct circ_buf *xmit;
	xmit = &port->state->xmit;

	/* Enable FIFO and Reset */
	nvt_write(port, UART_FCR_REG, FIFO_EN_BIT | RX_FIFO_RESET_BIT | TX_FIFO_RESET_BIT);
	/* Disable FIFO */
	nvt_write(port, UART_FCR_REG, 0);


	if(xmit->head != xmit->tail){
		serial_dbg("\nFIFO remain %d\n",CIRC_CNT_TO_END(xmit->head, xmit->tail, UART_XMIT_SIZE));

		/* Force reset FIFO buffer */
		xmit->head = xmit->tail;
	}
}

static void nvt_enable_rx_dma(struct uart_port* port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);

	serial_flow_dbg();
	serial_dbg("nvt_port->pin_pon_flag = %d\r\n",nvt_port->pin_pon_flag);

	if(nvt_port->pin_pon_flag == 1){
		nvt_write(port, UART_RX_DMA_ADDR_REG, nvt_port->rx_dma_addr1);
#if defined(UART_HIGH_ADDR_MASK) && defined(UART_RX_DMA_ADDR_H_REG)
		nvt_write(port, UART_RX_DMA_ADDR_H_REG, nvt_port->rx_dma_addr1 >> 32);
#endif
	} else {
		nvt_write(port, UART_RX_DMA_ADDR_REG, nvt_port->rx_dma_addr2);
#if defined(UART_HIGH_ADDR_MASK) && defined(UART_RX_DMA_ADDR_H_REG)
		nvt_write(port, UART_RX_DMA_ADDR_H_REG, nvt_port->rx_dma_addr2 >> 32);
#endif
	}

	nvt_write(port, UART_RX_DMA_SIZE_REG, nvt_port->rx_size);
	nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_DONE_INTEN_BIT, RX_DMA_DONE_INTEN_BIT);
	nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_ERR_INTEN_BIT, RX_DMA_ERR_INTEN_BIT);
	nvt_masked_write(port, UART_RX_DMA_CTRL_REG, RX_DMA_EN_BIT, RX_DMA_EN_BIT);
}

static void nvt_enable_tx_dma(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	struct circ_buf *xmit = &port->state->xmit; 
	int count = 0;

	serial_flow_dbg();

	nvt_port->tx_size = 0;
	count = CIRC_CNT_TO_END(xmit->head, xmit->tail, UART_XMIT_SIZE);

	/* tx send buffer must smaller than circ_buf */
	/* pin buffer DMA send */
	if(count > 0){
		nvt_port->tx_dma_addr = dma_map_single(port->dev, &xmit->buf[xmit->tail], UART_XMIT_SIZE, DMA_TO_DEVICE);
		dma_sync_single_for_device(port->dev, nvt_port->tx_dma_addr, UART_XMIT_SIZE, DMA_TO_DEVICE);
		xmit->tail = 0;
		xmit->head = xmit->tail;
		port->icount.tx += count;
		nvt_port->tx_size += count;

		nvt_write(port, UART_TX_DMA_ADDR_REG, nvt_port->tx_dma_addr);
		nvt_write(port, UART_TX_DMA_SIZE_REG, count);

#if defined(UART_HIGH_ADDR_MASK) && defined(UART_TX_DMA_ADDR_H_REG)
		nvt_write(port, UART_TX_DMA_ADDR_H_REG, nvt_port->tx_dma_addr >> 32);
#endif
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, TX_DMA_DONE_INTEN_BIT, TX_DMA_DONE_INTEN_BIT);
		nvt_masked_write(port, UART_TX_DMA_CTRL_REG, TX_DMA_EN_BIT, TX_DMA_EN_BIT);

		serial_dbg("TxData send %d byte to layer\n", count);

	}
}

static void nvt_handle_rx_dma(unsigned long data)
{
	unsigned long flags;
	struct nvt_port *nvt_port = (struct nvt_port *)data;
	struct uart_port *port = &nvt_port->uart;
	unsigned int lsr_r = nvt_port->lsr;
	struct tty_port *tport = &port->state->port;
	char flag = TTY_NORMAL;
	unsigned int copied, dma_transfered = 0;

	serial_flow_dbg();
	spin_lock_irqsave(&port->lock, flags);
	dma_transfered = nvt_port->rx_size;

	lsr_r |= nvt_port->lsr_break_flag;
	nvt_port->lsr_break_flag = 0;

	if (lsr_r & BREAK_INT_BIT) {
		port->icount.brk++;
		if (uart_handle_break(port))
			 goto dma_ignore;
	} else if (lsr_r & PARITY_ERR_BIT)
		port->icount.parity++;
	else if (lsr_r & FRAMING_ERR_BIT)
		port->icount.frame++;

	if (lsr_r & OVERRUN_ERR_BIT)
		port->icount.overrun++;
	
	/*
	 * Mask off conditions which should be ignored.
	 */
	lsr_r &= port->read_status_mask;

	if (lsr_r & BREAK_INT_BIT)
		flag = TTY_BREAK;
	else if (lsr_r & PARITY_ERR_BIT)
		flag = TTY_PARITY;
	else if (lsr_r & FRAMING_ERR_BIT)
		flag = TTY_FRAME;
	
	if(nvt_port->pin_pon_flag == 0) {
		copied = tty_insert_flip_string(tport, nvt_port->rx_virt_addr1, dma_transfered);
		nvt_port->pin_pon_flag = 1;
	} else {
		copied = tty_insert_flip_string(tport, nvt_port->rx_virt_addr2, dma_transfered);
		nvt_port->pin_pon_flag = 0;
	}

	if(copied != dma_transfered){
		serial_dbg("RxData copy to tty layer failed\n");
		port->icount.rx += copied;
	}else{
		serial_dbg("RxData copy %d byte to layer\n", dma_transfered);
		port->icount.rx += dma_transfered;
	}
	tty_flip_buffer_push(tport);

	/* Enanle receive timeout interrupts */
	nvt_write(port, UART_IER_REG, RDA_INTEN_BIT);
	
dma_ignore:
	tty_flip_buffer_push(tport);
	/* Enanle receive timeout interrupts */
	nvt_write(port, UART_IER_REG, RDA_INTEN_BIT);
	spin_unlock_irqrestore(&port->lock, flags);
}

static void nvt_flush_rx(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	struct tty_port *tport = &port->state->port;
	unsigned char ch;
	char flag;
	unsigned int lsr_r;

	serial_flow_dbg();

	while ((lsr_r = nvt_read(port, UART_LSR_REG)) & DATA_READY_BIT) {
		ch = nvt_read(port, UART_RBR_REG);

		flag = TTY_NORMAL;
		port->icount.rx++;

		serial_dbg("LSR(0x%x) cnt_rx(%d), read ch(0x%x) from RBR\n", lsr_r, port->icount.rx, ch);

		lsr_r |= nvt_port->lsr_break_flag;
		nvt_port->lsr_break_flag = 0;

		if (lsr_r & BREAK_INT_BIT) {
			port->icount.brk++;
			if (uart_handle_break(port))
				continue;
		} else if (lsr_r & PARITY_ERR_BIT)
			port->icount.parity++;
		else if (lsr_r & FRAMING_ERR_BIT)
			port->icount.frame++;

		if (lsr_r & OVERRUN_ERR_BIT)
			port->icount.overrun++;

		/*
		 * Mask off conditions which should be ignored.
		 */
		lsr_r &= port->read_status_mask;

		if (lsr_r & BREAK_INT_BIT)
			flag = TTY_BREAK;
		else if (lsr_r & PARITY_ERR_BIT)
			flag = TTY_PARITY;
		else if (lsr_r & FRAMING_ERR_BIT)
			flag = TTY_FRAME;

		if (!(uart_handle_sysrq_char(port, ch)))
			uart_insert_char(port, lsr_r, OVERRUN_ERR_BIT, ch, flag);
	}

	/*
	 * Drop the lock here since it might end up calling
	 * uart_start(), which takes the lock.
	 */
	spin_unlock(&port->lock);
	tty_flip_buffer_push(tport);
	spin_lock(&port->lock);
}

static void nvt_stop_rx(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);

	serial_flow_dbg();

	if (nvt_port->rx_dma_en && port->line != 0) {
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_DONE_INTEN_BIT, 0);
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_ERR_INTEN_BIT, 0);

		nvt_masked_write(port, UART_RX_DMA_CTRL_REG, RX_DMA_EN_BIT, 0);
		while (nvt_read(port, UART_RX_DMA_CTRL_REG) & RX_DMA_EN_BIT) {
			;
		}

		nvt_flush_rx(port);
	#if defined(CONFIG_SERIAL_NVT_DMA_TIMEOUT) && !defined(RX_DMA_PINPON_EN_BIT)
		del_timer_sync(&nvt_port->rx_timeout_timer);
	#endif
	}

	nvt_masked_write(port, UART_IER_REG, RLS_INTEN_BIT | RDA_INTEN_BIT, 0);
}

static void nvt_stop_tx(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);

	serial_flow_dbg();

	if (nvt_port->tx_dma_en && port->line != 0) {
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, TX_DMA_DONE_INTEN_BIT, 0);
		nvt_masked_write(port, UART_TX_DMA_CTRL_REG, TX_DMA_EN_BIT, 0);
		while (nvt_read(port, UART_TX_DMA_CTRL_REG) & TX_DMA_EN_BIT) {
			;
		}

		dma_unmap_single(port->dev, nvt_port->tx_dma_addr, UART_XMIT_SIZE, DMA_TO_DEVICE);
	}

	nvt_masked_write(port, UART_IER_REG, THR_EMPTY_INTEN_BIT, 0);
	while (nvt_read(port, UART_IER_REG) & THR_EMPTY_INTEN_BIT) {
		;
	}
}

#define UART_LSR_BRK_ERROR_BITS    0x1E /* BI, FE, PE, OE bits */
#define LSR_SAVE_FLAGS             UART_LSR_BRK_ERROR_BITS
static void nvt_handle_tx(struct uart_port *port);
static void nvt_start_tx(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);

	serial_flow_dbg();

	/*
	 * We can't control the number of start_tx in one startup cycle,
	 * which is determined by the number of SyS_write calls.
	 *
	 * If handle_tx exhausted the xmit buffer, it needs to be returned
	 * when xmit buffer is empty to prevent start_tx from hanging without
	 * data.
	 */
	if (uart_circ_empty(&port->state->xmit) || uart_tx_stopped(port)) {
		nvt_stop_tx(port);
		serial_dbg("uart_circ_empty or uart_tx_stopped, so return\n");
		return;
	}

	if (nvt_port->tx_dma_en && port->line != 0) {

		nvt_enable_tx_dma(port);
		while (nvt_read(port, UART_TX_DMA_CTRL_REG) & TX_DMA_EN_BIT) {
			serial_dbg("wait TX_DMA_EN_BIT auto clear\n");
			;
		}
	} else {
		nvt_masked_write(port, UART_IER_REG, THR_EMPTY_INTEN_BIT, THR_EMPTY_INTEN_BIT);
		while (!(nvt_read(port, UART_IER_REG) & THR_EMPTY_INTEN_BIT)) {
			;
		}
	}
}

static unsigned int nvt_tx_empty(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	unsigned int lsr_r;
	unsigned long flags;

	serial_flow_dbg();

	spin_lock_irqsave(&port->lock, flags);
	lsr_r = nvt_read(port, UART_LSR_REG);
	nvt_port->lsr_saved_flags |= lsr_r & LSR_SAVE_FLAGS;
	spin_unlock_irqrestore(&port->lock, flags);

	return (lsr_r & BOTH_EMPTY_BIT) ? TIOCSER_TEMT : 0;
}

static void nvt_handle_rx(struct uart_port *port, unsigned int lsr_r)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	struct tty_port *tport = &port->state->port;
	unsigned char ch;
	char flag;

	serial_flow_dbg();

	do {
		if (likely(lsr_r & DATA_READY_BIT))
			ch = nvt_read(port, UART_RBR_REG);
		else
			ch = 0;

		flag = TTY_NORMAL;
		port->icount.rx++;

		serial_dbg("LSR(0x%x) IER(0x%x) cnt_rx(%d), read ch(0x%x) from RBR\n", lsr_r, nvt_read(port, UART_IER_REG), port->icount.rx, ch);

		lsr_r |= nvt_port->lsr_saved_flags;
		nvt_port->lsr_saved_flags = 0;

		lsr_r |= nvt_port->lsr_break_flag;
		nvt_port->lsr_break_flag = 0;

		if (lsr_r & BREAK_INT_BIT) {
			port->icount.brk++;
			if (uart_handle_break(port))
				continue;
		} else if (lsr_r & PARITY_ERR_BIT)
			port->icount.parity++;
		else if (lsr_r & FRAMING_ERR_BIT)
			port->icount.frame++;

		if (lsr_r & OVERRUN_ERR_BIT)
			port->icount.overrun++;

		/*
		 * Mask off conditions which should be ignored.
		 */
		lsr_r &= port->read_status_mask;

		if (lsr_r & BREAK_INT_BIT)
			flag = TTY_BREAK;
		else if (lsr_r & PARITY_ERR_BIT)
			flag = TTY_PARITY;
		else if (lsr_r & FRAMING_ERR_BIT)
			flag = TTY_FRAME;

		if (!(uart_handle_sysrq_char(port, ch)))
			uart_insert_char(port, lsr_r, OVERRUN_ERR_BIT, ch, flag);
	} while ((lsr_r = nvt_read(port, UART_LSR_REG)) & DATA_READY_BIT);

	tty_flip_buffer_push(tport);
}

#ifdef AB_MODIFIED

static uint8_t iab_logdump_queue_wait_count(void)
{
    uint8_t uc_ret = 0;

#if AB_LOGDUMP_EN_KFIFO
    if(kfifo_initialized(&s_transfer_queue) != false){
        uc_ret = (uint8_t)(kfifo_len(&s_transfer_queue)/sizeof(AB_LOGDUMP_BUFF_ST*));
    }

#else
    if(s_transfer_info.c_output_idx != s_transfer_info.c_input_idx) {
        uc_ret = 1;
    }
    
#endif
    return uc_ret;
}

static int32_t iab_logdump_pop_queue(AB_LOGDUMP_BUFF_ST** pps_buffer)
{
    int32_t i_ret = 0;

#if AB_LOGDUMP_EN_KFIFO
    if(kfifo_initialized(&s_transfer_queue) != false){
        if(kfifo_out(&s_transfer_queue, pps_buffer, sizeof(AB_LOGDUMP_BUFF_ST*)) != sizeof(AB_LOGDUMP_BUFF_ST*)) {
            i_ret = -1;
        }
        else {
            i_ret = 0;
        }
    }
    else {
        i_ret = -1;
    }

#else
    int8_t c_target_idx = 0;

    c_target_idx = (s_transfer_info.c_output_idx+1) % AB_LOGDUMP_BUFF_MAX_CNT;
    if(ps_transfer_queue[c_target_idx] == NULL) {
        i_ret = -1;
    }
    else {
        *pps_buffer = ps_transfer_queue[c_target_idx];
        ps_transfer_queue[c_target_idx] = NULL;
        s_transfer_info.c_output_idx = (s_transfer_info.c_output_idx+1) % (AB_LOGDUMP_BUFF_MAX_CNT*2);
    }

#endif
    return i_ret;
}

static int32_t iab_logdump_push_queue(AB_LOGDUMP_BUFF_ST** pps_buffer)
{
    int32_t i_ret = 0;

#if AB_LOGDUMP_EN_KFIFO
    if(kfifo_initialized(&s_transfer_queue) != false){
        if(kfifo_in(&s_transfer_queue, pps_buffer, sizeof(AB_LOGDUMP_BUFF_ST*)) != sizeof(AB_LOGDUMP_BUFF_ST*)) {
            i_ret = -1;
        }
        else {
            i_ret = 0;
        }
    }
    else {
        i_ret = -1;
    }

#else
    int8_t c_target_idx = 0;

    c_target_idx = (s_transfer_info.c_input_idx+1) % AB_LOGDUMP_BUFF_MAX_CNT;
    if(ps_transfer_queue[c_target_idx] != NULL) {
        i_ret = -1;
    }
    else {
        ps_transfer_queue[c_target_idx] = *pps_buffer;
        s_transfer_info.c_input_idx = (s_transfer_info.c_input_idx+1) % (AB_LOGDUMP_BUFF_MAX_CNT*2);
    }

#endif
    return i_ret;
}

static bool iab_logdump_is_all_main_buffer_valid(void)
{
    bool b_ret = true;
    uint8_t uc_idx = 0;
    AB_LOGDUMP_BUFF_ST *ps_buff = NULL;
    for(uc_idx = 0; uc_idx<AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM; uc_idx++) {
        if(uc_idx == 0) {
            ps_buff = ps_logbuff;
        }

        if(ps_buff->uc_bufstatus != AB_LOGDUMP_BUFF_VALID) {
            b_ret = false;
            break;// break for loop
        }

        ps_buff = (AB_LOGDUMP_BUFF_ST *)ps_buff->ul_next_bufptr;
    }
    return b_ret;
}

static bool iab_logdump_is_all_main_buffer_valid_and_inuse(void)
{
    bool b_ret = true;
    uint8_t uc_idx = 0, uc_valid_cnt = 0, uc_inuse_cnt = 0;
    AB_LOGDUMP_BUFF_ST *ps_buff = NULL;
	
	
    for(uc_idx = 0; uc_idx<AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM; uc_idx++) {
        if(uc_idx == 0) {
            ps_buff = ps_logbuff;
        }

        if(ps_buff->uc_bufstatus != AB_LOGDUMP_BUFF_VALID && ps_buff->uc_bufstatus != AB_LOGDUMP_BUFF_IN_USE) {
            b_ret = false;
            break;// break for loop
        }
        else {
            if(ps_buff->uc_bufstatus == AB_LOGDUMP_BUFF_IN_USE) {
                uc_inuse_cnt++;
            }
            else if(ps_buff->uc_bufstatus == AB_LOGDUMP_BUFF_VALID) {
                uc_valid_cnt++;
            }
        }

        ps_buff = (AB_LOGDUMP_BUFF_ST *)ps_buff->ul_next_bufptr;
    }

    if(uc_inuse_cnt == 0 ||  (uc_inuse_cnt+uc_valid_cnt) != AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM) {
        b_ret = false;
    }
	
    return b_ret;

}

static bool iab_logdump_is_current_use_dynamic_buffer(void)
{
    bool b_ret = true;
    uint8_t uc_idx = 0;
    AB_LOGDUMP_BUFF_ST *ps_buff = NULL;
	
	
    for(uc_idx = 0; uc_idx<AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM; uc_idx++) {
        if(uc_idx == 0) {
            ps_buff = ps_logbuff;
        }

        if(ps_curlogbuff == ps_buff) {
            b_ret = false;
            break;// break for loop
        }

        ps_buff = (AB_LOGDUMP_BUFF_ST *)ps_buff->ul_next_bufptr;
    }
	
    return b_ret;
}

//static uint64_t iab_logdump_get_last_struct_addr(AB_LOGDUMP_POOL_ET e_pool)
static void* iab_logdump_get_last_struct_addr(AB_LOGDUMP_POOL_ET e_pool)
{
    //uint64_t ul_addr = 0;
    void* ul_addr = NULL;
	uint8_t uc_idx = 0;
    AB_LOGDUMP_BUFF_ST* ps_buff = NULL;
	

    
    if(e_pool == AB_LOGDUMP_POOL_MAIN) {
        for(uc_idx = 0; uc_idx < AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM; uc_idx++) {
            if(uc_idx == 0) {
                ps_buff = (AB_LOGDUMP_BUFF_ST*)pv_main_buff_pool;
            }
            else {
                ps_buff = (AB_LOGDUMP_BUFF_ST*)ps_buff->ul_next_bufptr;
            }
        }
    }
    else if(e_pool == AB_LOGDUMP_POOL_DYNAMIC) {
        if(pv_dynamic_buff_pool != NULL) {
            for(uc_idx = 0; uc_idx < AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM; uc_idx++) {
                if(uc_idx == 0) {
                    ps_buff = (AB_LOGDUMP_BUFF_ST*)pv_dynamic_buff_pool;
                }
                else {
                    ps_buff = (AB_LOGDUMP_BUFF_ST*)ps_buff->ul_next_bufptr;
                }
            }
        }
    }

    //ul_addr = (uint64_t)ps_buff;
	ul_addr = ps_buff;
    
    return ul_addr;
}

static int32_t iab_logdump_switch_logbuff(AB_LOGDUMP_SWITCH_BUFF_MECH_ET e_mechanism)
{
    switch(e_mechanism) {
        default:
        case AB_LOGDUMP_SWITCH_BUFF_MECH_SEQUENCE:
            {
                AB_LOGDUMP_BUFF_ST* ps_target_buff = NULL;

                // step 1: push buffer to transfer queue
                if(ps_curlogbuff->uc_inqueue == false){
                    if(iab_logdump_push_queue(&ps_curlogbuff)) {
                        AB_LOGDUMP_PRINT(true, "%s push logbuff to transfer queue fail\n", __FUNCTION__);
                        return -1;
                    }
                    ps_curlogbuff->uc_inqueue = true;
                }

                // step 2: switch to next buffer
                ps_target_buff = (AB_LOGDUMP_BUFF_ST*)ps_curlogbuff->ul_next_bufptr;
                if(ps_target_buff->uc_bufstatus == AB_LOGDUMP_BUFF_VALID) {
                    ps_curlogbuff = ps_target_buff;
                    ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_IN_USE;
                    ps_curlogbuff->uw_datalen   = 0;
                    ps_curlogbuff->uc_inqueue   = false;
                    memset(ps_curlogbuff->uc_data, 0, sizeof(ps_curlogbuff->uc_data));  
                }
                else {
                    AB_LOGDUMP_PRINT(true, "%s switch to next log buffer fail!\n", __FUNCTION__);
                    return -1;
                }
                
                AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "%s mechanism(%d), queue(%d)\n", __FUNCTION__, e_mechanism, iab_logdump_queue_wait_count());
            }
            break;

        case AB_LOGDUMP_SWITCH_BUFF_MECH_REDIRECT_TOP:
            {
                AB_LOGDUMP_BUFF_ST* ps_target_buff = NULL;
				void* ul_buff_addr = NULL;
                //uint64_t ul_buff_addr = 0;

                // step 1: push buffer to transfer queue
                if(ps_curlogbuff->uc_inqueue == false) {
                    if(iab_logdump_push_queue(&ps_curlogbuff)) {
                        AB_LOGDUMP_PRINT(true, "%s push logbuff to transfer queue fail\n", __FUNCTION__);
                        return -1;
                    }
                    ps_curlogbuff->uc_inqueue = true;
                }

                // step 2: double confirm main buffer pool's buffers are all valid and switch to buffer 0
                if(iab_logdump_is_all_main_buffer_valid() == false) {
                    AB_LOGDUMP_PRINT(true, "%s force switch to buffer 0 fail\n", __FUNCTION__);
                    return -1;
                }

                ps_curlogbuff = ps_logbuff;
                ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_IN_USE;
                ps_curlogbuff->uw_datalen   = 0;
                ps_curlogbuff->uc_inqueue   = false;
                memset(ps_curlogbuff->uc_data, 0, sizeof(ps_curlogbuff->uc_data));  
                
                // step 3: reset last main buffer's next buffer addr
                ul_buff_addr = iab_logdump_get_last_struct_addr(AB_LOGDUMP_POOL_MAIN);
                if(ul_buff_addr == NULL) {
				//if(ul_buff_addr == 0) {
                    AB_LOGDUMP_PRINT(true, "%s reset last main buffer's next buffer addr fail!!\n", __FUNCTION__);
                    return -1;                    
                }
                ps_target_buff = (AB_LOGDUMP_BUFF_ST*)ul_buff_addr;
				//ps_target_buff->ul_next_bufptr = (uint64_t)ps_logbuff;
                ps_target_buff->ul_next_bufptr = ps_logbuff;
                AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "%s mechanism(%d), queue(%d)\n", __FUNCTION__, e_mechanism, iab_logdump_queue_wait_count());
            }
            break;

        case AB_LOGDUMP_SWITCH_BUFF_MECH_OVERWRITE_SEQ:
            {
                AB_LOGDUMP_BUFF_ST* ps_target_buff = NULL;
                AB_LOGDUMP_PRINT(false, "%s, mech %d s\n", __FUNCTION__, e_mechanism);
                // WARN: There have 1 risk, When all dynamic buffers are all in queue to transfer and main buffer 0 also in queue.
                //       If buffer 1 full, the oldest buffer is  dynamic buffer, it will use dynamic buffer as next buffer.
                //       If this condition occur, need to fix it .
                // step 1: pop oldest buffer
                if(iab_logdump_pop_queue(&ps_target_buff)) {
                    AB_LOGDUMP_PRINT(true, "%s pop oldest logbuffer fail\n", __FUNCTION__);
                    return -1;
                }
                
                // step 2: push buffer to trnasfer queue
                if(ps_curlogbuff->uc_inqueue == false) {
                    if(iab_logdump_push_queue(&ps_curlogbuff)) {
                        AB_LOGDUMP_PRINT(true, "%s push logbuff to transfer queue fail\n", __FUNCTION__);
                        return -1;
                    }
                    ps_curlogbuff->uc_inqueue = true;
                }

                // step 3: assign to next buffer
                ps_curlogbuff = ps_target_buff;
                ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_IN_USE;
                ps_curlogbuff->uw_datalen   = 0;
                ps_curlogbuff->uc_inqueue   = false;
                memset(ps_curlogbuff->uc_data, 0, sizeof(ps_curlogbuff->uc_data));

                AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "%s mechanism(%d), queue(%d)\n", __FUNCTION__, e_mechanism, iab_logdump_queue_wait_count());
            }
            break;

        case AB_LOGDUMP_SWITCH_BUFF_MECH_DROP_SEQ:
            {
                if(ps_curlogbuff->uc_inqueue == false) {
                    // step 2: push buffer to trnasfer queue
                    if(iab_logdump_push_queue(&ps_curlogbuff)) {
                        AB_LOGDUMP_PRINT(true, "%s push logbuff to transfer queue fail\n", __FUNCTION__);
                        return -1;
                    }
                    ps_curlogbuff->uc_inqueue = true;
                }
                else {
                    s_buff_info.ul_drop_log_len++;
                }                
            }
            break;
            
    }

    return 0;
}

static bool iab_logdump_is_allbufferfull(void)
{
    uint8_t uc_idx = 0;
    AB_LOGDUMP_BUFF_ST *ps_buff = NULL;

    ps_buff = ps_curlogbuff;
    for(uc_idx = 0; uc_idx<s_buff_info.uc_buff_max_idx; uc_idx++) {
        if((ps_buff->uc_bufstatus == AB_LOGDUMP_BUFF_VALID)) {
            // check not all buffer full
            return false;
        }
        else {
            ps_buff = (AB_LOGDUMP_BUFF_ST *)ps_buff->ul_next_bufptr;
        }
    }

    return true;
}

static void iab_logdump_switch_logbuff_default_action(void)
{
    bool b_enable_switch_next_buff = false;

    if(ps_curlogbuff->uw_datalen > 0 && ps_curlogbuff->uw_datalen <= AB_LOGDUMP_BUFF_DATA_MAXSIZE) {
        if((ps_curlogbuff->uc_data[ps_curlogbuff->uw_datalen-1] == '\n') /*|| (ps_curlogbuff->uc_data[ps_curlogbuff->uw_datalen-1] == '\r')*/) {
            s_buff_info.b_is_occur_linebreak = true;
        }
    }

    AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "ab_log uw_datalen(%d) uc_bufstatus(%d) b_is_occur_linebreak(%d) b_logdump_forcedump_trig(%d)\n", ps_curlogbuff->uw_datalen, ps_curlogbuff->uc_bufstatus, s_buff_info.b_is_occur_linebreak, ps_desc->b_logdump_forcedump_trig);

    if(ps_curlogbuff->uw_datalen >= AB_LOGDUMP_BUFF_DATA_MAXSIZE || ps_curlogbuff->uc_bufstatus == AB_LOGDUMP_BUFF_FULL) { // remain size is equal 0
        b_enable_switch_next_buff = true;
    }
    else if((ps_curlogbuff->uw_datalen > (AB_LOGDUMP_BUFF_DATA_MAXSIZE-AB_LOGDUMP_BUFF_REMAIN_LIMIT_SIZE)) &&
             (s_buff_info.b_is_occur_linebreak == true)) {  // remain size < 256 and current character is line break
        b_enable_switch_next_buff = true;
    }
    else if(ps_desc->b_logdump_forcedump_trig == true || ps_desc->b_logdump_terminate_trig == true) {    // force dump flag enable
        b_enable_switch_next_buff = true;
        if(ps_desc->b_logdump_forcedump_trig == true) 
            ps_desc->b_logdump_forcedump_trig = false;
    }

    if(b_enable_switch_next_buff == true) {
        bool b_is_allbuffer_full = false;

        // set current buffer status as full
        if(ps_curlogbuff->uc_bufstatus == AB_LOGDUMP_BUFF_IN_USE)
            ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_FULL;
        
        // check all buffer's status
        b_is_allbuffer_full = iab_logdump_is_allbufferfull();
        
        if(b_is_allbuffer_full == true) { // all buffers are full
            if(s_buff_info.b_logbuff_en_overwrite)
                iab_logdump_switch_logbuff(AB_LOGDUMP_SWITCH_BUFF_MECH_OVERWRITE_SEQ);
            else 
                iab_logdump_switch_logbuff(AB_LOGDUMP_SWITCH_BUFF_MECH_DROP_SEQ);
        }
        else {
            iab_logdump_switch_logbuff(AB_LOGDUMP_SWITCH_BUFF_MECH_SEQUENCE);
        }
    }
}

static void iab_logdump_fill_infolog(AB_LOGDUMP_FILL_INFOLOG_ET e_info_type)
{
    uint16_t uw_copy_len = 0;
    char c_initloginfo[AB_LOGDUMP_INITLOG_MAX_LEN] = "\0";

    if(ps_curlogbuff->uw_datalen < AB_LOGDUMP_BUFF_DATA_MAXSIZE) {
        switch(e_info_type) {
            case AB_LOGDUMP_FILL_INFOLOG_INIT_INFO:
                {
                    snprintf(c_initloginfo, sizeof(c_initloginfo)-1, "==== HOST LOG DUMP ====\n");

                    if(strlen(c_initloginfo) > (AB_LOGDUMP_BUFF_DATA_MAXSIZE-ps_curlogbuff->uw_datalen)) {
                        uw_copy_len = (uint16_t)(AB_LOGDUMP_BUFF_DATA_MAXSIZE-ps_curlogbuff->uw_datalen);
                    }
                    else {
                        uw_copy_len = (uint16_t)strlen(c_initloginfo);
                    }
                }
                break;
            
            case AB_LOGDUMP_FILL_INFOLOG_DROP_INFO:
                {
                    snprintf(c_initloginfo, sizeof(c_initloginfo)-1, "## Drop log %lld bytes ##\n", s_buff_info.ul_drop_log_len);
                    AB_LOGDUMP_PRINT(true, "## Drop log %lld bytes ##\n", s_buff_info.ul_drop_log_len);
                    
                    if(strlen(c_initloginfo) > (AB_LOGDUMP_BUFF_DATA_MAXSIZE-ps_curlogbuff->uw_datalen)) {
                        uw_copy_len = (uint16_t)(AB_LOGDUMP_BUFF_DATA_MAXSIZE-ps_curlogbuff->uw_datalen);
                    }
                    else {
                        uw_copy_len = (uint16_t)strlen(c_initloginfo);
                    }
                }
                break;
            
            case AB_LOGDUMP_FILL_INFOLOG_TIMESTAMP_INFO:
                {
                    struct timespec64 s_ts;

                    _get_boottime(&s_ts);

                    if(ps_desc->e_timstamp_accuracy == AB_LODDUMP_TIMESTAMP_ACCURACY_MS) {
                        snprintf(c_initloginfo, sizeof(c_initloginfo)-1, "[%*lld.%.*ld] ", AB_LOGDUMP_TIMESTAMP_PRE_LEN, (s_ts.tv_sec-ps_desc->s_ts.tv_sec), AB_LOGDUMP_TIMESTAMP_POST_MS_LEN, s_ts.tv_nsec/1000000);
                    }
                    else {
                        snprintf(c_initloginfo, sizeof(c_initloginfo)-1, "[%*lld.%.*ld] ", AB_LOGDUMP_TIMESTAMP_PRE_LEN, (s_ts.tv_sec-ps_desc->s_ts.tv_sec), AB_LOGDUMP_TIMESTAMP_POST_US_LEN, s_ts.tv_nsec/1000);
                    }

                    if(strlen(c_initloginfo) > (AB_LOGDUMP_BUFF_DATA_MAXSIZE-ps_curlogbuff->uw_datalen)) {
                        uw_copy_len = (uint16_t)(AB_LOGDUMP_BUFF_DATA_MAXSIZE-ps_curlogbuff->uw_datalen);
                    }
                    else {
                        uw_copy_len = (uint16_t)strlen(c_initloginfo);
                    }
                }
                break;
            
            default:
                return;
                break;
        }

        if(uw_copy_len) {
			

    		
            memcpy(&ps_curlogbuff->uc_data[ps_curlogbuff->uw_datalen], (void*)c_initloginfo, uw_copy_len);
            ps_curlogbuff->uw_datalen += uw_copy_len;
			
        }
    }
}

static void ab_logdump_serial_dump(int value)
{
    uint8_t uc_transfer_queue_num = 0;
    bool b_uartinit_ready = false;
    unsigned long flags = 0;

    // Step 0: make sure service is init
    if(ps_curlogbuff == NULL || ps_desc == NULL) {
        return ;
    }

    if(ps_desc->b_logdump_buff_ready == false) return ;

    b_uartinit_ready = ps_desc->b_uartinit_ready;
    spin_lock_irqsave(&ab_logdump_lock, flags);
    // Step 0-1: all valid buffer case, assign inuse buffer
    if(ps_curlogbuff->uc_bufstatus == AB_LOGDUMP_BUFF_VALID) {
        if( ps_curlogbuff->uw_datalen != 0 ) {
            AB_LOGDUMP_PRINT(true, "ab_log err!! valid buffer with non-zero data length (%d)\n", ps_curlogbuff->uw_datalen);
            ps_curlogbuff->uw_datalen = 0;
        }
        ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_IN_USE;
    }

    // Step 1: check if all buffer is full.
    //         Note: Fill character first and then switch buffer. Thus, if current buffer is full, it mean all buffers are full.
    if(ps_curlogbuff->uc_bufstatus == AB_LOGDUMP_BUFF_IN_USE) {
        // 1-0: fill initial log information at the beginning
        if(ps_desc->b_logdump_fillheader == false) {
            iab_logdump_fill_infolog(AB_LOGDUMP_FILL_INFOLOG_INIT_INFO);
            ps_desc->b_logdump_fillheader = true;
        }

        // 1-0.1: fill drop info
        if(s_buff_info.ul_drop_log_len) {
            iab_logdump_fill_infolog(AB_LOGDUMP_FILL_INFOLOG_DROP_INFO);
            s_buff_info.ul_drop_log_len = 0;
        }

        // 1-1 check if need to fill timestamp or not and fill timestamp
        // 1-1.1 fill timestamp
        if(s_buff_info.b_is_occur_linebreak == true) {
            iab_logdump_fill_infolog(AB_LOGDUMP_FILL_INFOLOG_TIMESTAMP_INFO);
            s_buff_info.b_is_occur_linebreak = false;
        }

        // 1-1.2 fill character, also need to make sure it will not overflow
        if(ps_curlogbuff->uw_datalen < AB_LOGDUMP_BUFF_DATA_MAXSIZE ){
			
			
            ps_curlogbuff->uc_data[ps_curlogbuff->uw_datalen] = value;

            if(ps_curlogbuff->uc_data[ps_curlogbuff->uw_datalen] == '\r') {
                ps_curlogbuff->uc_data[ps_curlogbuff->uw_datalen] = ' ';
            }
        
            ps_curlogbuff->uw_datalen++;
			
        }
    }
    else {  /* invalid and full case */
        // Note:
        // [drop mechanism] occur buff full case, it mean direct drop
        // [overwrite mechanism] it must not occur buffer full case, it should be switch to overwrite buffer at previous buffer check process.
        // WARNING: 
        // if current buffer is not IN_USE or FULL status, it may have something wrong, need check.
    }

    // Step 2: Do buffer check and switch buffer
    if(b_uartinit_ready == true && pv_dynamic_buff_pool != NULL) {
        bool b_is_allmainbuffer_valid = iab_logdump_is_all_main_buffer_valid();
        bool b_is_currbuff_dynamic_pool = iab_logdump_is_current_use_dynamic_buffer();
    
        AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "ab_log b_uartinit_ready(%d) dynamic_pool(%p) b_is_allmainbuffer_valid(%d) b_is_currbuff_dynamic_pool(%d)\n", b_uartinit_ready, pv_dynamic_buff_pool, b_is_allmainbuffer_valid, b_is_currbuff_dynamic_pool);
        if(b_is_allmainbuffer_valid && b_is_currbuff_dynamic_pool) {
            ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_FULL;
            s_buff_info.uc_buff_max_idx = AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM;
            
            // ACTION: switch to buffer 0/1 and set current buffer status from inused to full.
            iab_logdump_switch_logbuff(AB_LOGDUMP_SWITCH_BUFF_MECH_REDIRECT_TOP);
        }
        else {
            AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "ab_log default action_a!!\n");
            // ACTION: go default process action.
            iab_logdump_switch_logbuff_default_action();
        }
    }
    else {
        AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "ab_log default action_b!!\n");
        // ACTION: go default process action.
        iab_logdump_switch_logbuff_default_action();
    }

    // Step 3: handle free dynamic buffer pool and terminate case
    // 3-1: check terminate flag and check is need to notify to transfer
    uc_transfer_queue_num = iab_logdump_queue_wait_count();
    AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "ab_log ps_open_info(%p) b_uartinit_ready(%d) uc_transfer_queue_num(%d)\n", ps_open_info, b_uartinit_ready, uc_transfer_queue_num);
    if(ps_open_info != NULL) {
        if(b_uartinit_ready == true && 
          (ps_desc->b_logdump_terminate_trig == true || uc_transfer_queue_num)) 
        {
            if(ps_open_info->b_wait_wakeup == true) {
			    unsigned long flags = 0;
                //down(&ps_open_info->s_protect_sem);
				spin_lock_irqsave(&ps_open_info->s_read_lock, flags);
                ps_open_info->b_wait_wakeup = false;
            #if AB_LOGDUMP_EN_TIME_MEASUREMENT
                _get_boottime(&ps_desc->s_measurement[1]);
                wake_up_interruptible(&ps_open_info->t_read_wait);
            #else
                AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "ab_log wake_up_interruptible!!\n");
                wake_up_interruptible(&ps_open_info->t_read_wait);
            #endif
                spin_unlock_irqrestore(&ps_open_info->s_read_lock, flags);
                //up(&ps_open_info->s_protect_sem);
            }
        }
    }

    // Step 4: handle dynamic buffer free
    if(b_uartinit_ready == true && pv_dynamic_buff_pool != NULL) {
        uint8_t uc_locked_cnt = 0;
    
        if(iab_logdump_is_all_main_buffer_valid_and_inuse()) 
        {
             AB_LOGDUMP_BUFF_ST *ps_cur_dynamic_buff = pv_dynamic_buff_pool;
             uint8_t uc_buff_idx = 0;
        
            // set dynamic buffer from valid to invalid
            for(uc_buff_idx = 0; uc_buff_idx<AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM ; uc_buff_idx++) {
                if(ps_cur_dynamic_buff->uc_bufstatus == AB_LOGDUMP_BUFF_VALID) {
                    ps_cur_dynamic_buff->uc_bufstatus = AB_LOGDUMP_BUFF_INVALID;
                }
                else if(ps_cur_dynamic_buff->uc_bufstatus == AB_LOGDUMP_BUFF_IN_USE || ps_cur_dynamic_buff->uc_bufstatus == AB_LOGDUMP_BUFF_FULL) {
                    uc_locked_cnt++;
                }
                ps_cur_dynamic_buff = (AB_LOGDUMP_BUFF_ST *)ps_cur_dynamic_buff->ul_next_bufptr;
            }
        }
        else {
            uc_locked_cnt = 1; // at least 1 buffer is locked
        }

        if(uc_locked_cnt == 0)   // mean all dynamic buffer is invalid
        {
            AB_LOGDUMP_BUFF_ST *ps_buff = NULL;

            AB_LOGDUMP_PRINT(true, "[ab_log] free dynamic pool(%d/%d)\n", AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM, AB_LOGDUMP_BUFF_MAX_CNT);
            // disable link 
            ps_buff = (AB_LOGDUMP_BUFF_ST *)iab_logdump_get_last_struct_addr(AB_LOGDUMP_POOL_MAIN);

			//ps_buff->ul_next_bufptr = (uint64_t)ps_logbuff;
			ps_buff->ul_next_bufptr = ps_logbuff;
            s_buff_info.uc_buff_max_idx = AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM;
            
            // free dynamic buffer
            kfree(pv_dynamic_buff_pool);
            pv_dynamic_buff_pool = NULL;
        }
    }
    spin_unlock_irqrestore(&ab_logdump_lock, flags);
}
#endif

static void nvt_handle_tx(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	struct circ_buf *xmit = &port->state->xmit;
	unsigned char ch;
	int count;

	serial_flow_dbg();

	/* For software flow control, xon resume transmission, xoff pause transmission */
	if (port->x_char) {
		nvt_write(port, UART_THR_REG, port->x_char);
#ifdef AB_MODIFIED
        /* Fill each character to ab_logdump's buffer */
        ab_logdump_serial_dump(port->x_char);
#endif
		port->icount.tx++;
		port->x_char = 0;
		return;
	}

	/* If there isn't anything more to transmit, or the uart is now stopped, disable the uart and exit */
	if (uart_circ_empty(xmit) || uart_tx_stopped(port)) {
		nvt_stop_tx(port);
		return;
	}

	/* Drain the buffer by size of tx_loadsz in one cycle */
	count = nvt_port->tx_loadsz;
	do {
		ch = xmit->buf[xmit->tail];
		nvt_write(port, UART_THR_REG, ch);
#ifdef AB_MODIFIED
        /* Fill each character to ab_logdump's buffer */
        ab_logdump_serial_dump(ch);
#endif
		port->icount.tx++;
		serial_dbg("head(%d) tail(%d) cnt_tx(%d), write ch(0x%x) to THR\n", xmit->head, xmit->tail, port->icount.tx, ch);
		xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
		if (uart_circ_empty(xmit))
			break;
	} while (--count > 0);

	/*
	 * If num chars in xmit buffer are too few, ask tty layer for more.
	 * By Hard ISR to schedule processing in software interrupt part.
	 */
	if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS) {
		uart_write_wakeup(port);
	}

	if (uart_circ_empty(xmit)) {
		nvt_stop_tx(port);
	}
}

static void rx_dma_timeout(unsigned long data)
{
	struct nvt_port *nvt_port = (struct nvt_port *)data;
	struct uart_port *port = &nvt_port->uart;
	struct tty_port *tport = &port->state->port;
	unsigned long  flags;

	serial_flow_dbg();

	spin_lock_irqsave(&nvt_port->rx_timeout_lock, flags);

	if (nvt_port->DMA_Counter > 0) {
		if(nvt_port->pin_pon_flag == 0) {
			serial_dbg("port->buf.tail = %lx nvt_port->rx_virt_addr1 = %lx",(uintptr_t)tport->buf.tail, (uintptr_t)nvt_port->rx_virt_addr1);
			tty_insert_flip_string(&port->state->port, nvt_port->rx_virt_addr1, nvt_port->DMA_Counter);
		} else {
			serial_dbg("port->buf.tail = %lx nvt_port->rx_virt_addr2 = %lx",(uintptr_t)tport->buf.tail, (uintptr_t)nvt_port->rx_virt_addr2);
			tty_insert_flip_string(&port->state->port, nvt_port->rx_virt_addr2, nvt_port->DMA_Counter);
		}
	}

	if (nvt_port->FIFO_Counter > 0) {
		tty_insert_flip_string(&port->state->port, nvt_port->ch, nvt_port->FIFO_Counter);
	}

	if (nvt_port->DMA_Counter > 0 || nvt_port->FIFO_Counter > 0) {
		tty_flip_buffer_push(&port->state->port);
		port->icount.rx = port->icount.rx + nvt_port->DMA_Counter + nvt_port->FIFO_Counter;
	}

	if (nvt_port->pin_pon_flag == 0)
		nvt_port->pin_pon_flag = 1;
	else
		nvt_port->pin_pon_flag = 0;
	
	/* Enanle receive timeout interrupts */
	nvt_write(port, UART_IER_REG, RDA_INTEN_BIT);

	spin_unlock_irqrestore(&nvt_port->rx_timeout_lock, flags);

}

#if defined(RX_DMA_PINPON_EN_BIT) 
static void nvt_rx_dma_timeout_get_pinpon_data(struct nvt_port *nvt_port)
{
	int dma_transfered = 0;	
	struct uart_port *port = &nvt_port->uart;
	struct tty_port *tport = &port->state->port;

	if (nvt_read(port, UART_RX_CUR_ADDR_REG) == nvt_read(port, UART_RX_DMA_ADDR_REG)) {
		dma_transfered = (nvt_read(port, UART_RX_CUR_ADDR2_REG) & RX_CUR_ADDRESS2_MASK) - 
			nvt_port->rx_dma_addr2;
		tty_insert_flip_string(&port->state->port, nvt_port->rx_virt_addr2, dma_transfered);
	} else {
		dma_transfered = (nvt_read(port, UART_RX_CUR_ADDR_REG) & RX_CUR_ADDRESS_MASK) - 
			nvt_port->rx_dma_addr1;
		tty_insert_flip_string(&port->state->port, nvt_port->rx_virt_addr1, dma_transfered);
	}

	nvt_port->DMA_Counter = dma_transfered;
	port->icount.rx = port->icount.rx + nvt_port->DMA_Counter;

	spin_unlock(&port->lock);
	tty_flip_buffer_push(tport);
	spin_lock(&port->lock);

	/* Enanle receive timeout interrupts */
	nvt_write(port, UART_IER_REG, RDA_INTEN_BIT);

}
#else
static void nvt_rx_dma_timeout_get_data(struct nvt_port *nvt_port)
{
	int dma_transfered = 0;	
	unsigned long lsr_r, flags;
	struct uart_port *port = &nvt_port->uart;

	spin_lock_irqsave(&nvt_port->rx_timeout_lock, flags);

	/* Handle rx dma timeout*/
	if(nvt_port->pin_pon_flag == 0) {
		dma_transfered = (nvt_read(port, UART_RX_CUR_ADDR_REG) & RX_CUR_ADDRESS_MASK) - 
			nvt_port->rx_dma_addr1;
	} else {
		dma_transfered = (nvt_read(port, UART_RX_CUR_ADDR_REG) & RX_CUR_ADDRESS_MASK) - 
			nvt_port->rx_dma_addr2;
	}

	// fire timeout tasklet
	if(dma_transfered >= 0 && dma_transfered < nvt_port->rx_size) {
		// Turn off RX DMA First
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_DONE_INTEN_BIT, 0);
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_ERR_INTEN_BIT, 0);
		nvt_masked_write(port, UART_RX_DMA_CTRL_REG, RX_DMA_EN_BIT, 0);

		// Get dma data
		nvt_port->DMA_Counter = dma_transfered;
		// Get remain data from UART FIFO
		dma_transfered = 0;
		while ((lsr_r = nvt_read(port, UART_LSR_REG)) & DATA_READY_BIT) {
			nvt_port->ch[dma_transfered] = nvt_read(port, UART_RBR_REG);
			dma_transfered++;
		}
		nvt_port->FIFO_Counter = dma_transfered;
		nvt_port->lsr = lsr_r;
		tasklet_schedule(&nvt_port->timeout_tasklet);
		nvt_enable_rx_dma(port);
	}

	spin_unlock_irqrestore(&nvt_port->rx_timeout_lock, flags);

#if defined(CONFIG_SERIAL_NVT_DMA_TIMEOUT)  && !defined(RX_DMA_PINPON_EN_BIT)
	mod_timer(&nvt_port->rx_timeout_timer, jiffies + msecs_to_jiffies(CONFIG_SERIAL_NVT_TIMEOUT_TIME));
#endif
}
#endif

#if defined(CONFIG_SERIAL_NVT_DMA_TIMEOUT) && !defined(RX_DMA_PINPON_EN_BIT)
static void nvt_rx_timeout_timer_handle( struct timer_list *t)
{
	unsigned long idle_r;
	struct nvt_port *nvt_port = from_timer(nvt_port, t, rx_timeout_timer);
	struct uart_port *port = &nvt_port->uart;
	
	idle_r =  nvt_read(port, UART_SPR_REG);

	if (idle_r & RX_IDLE_BIT) {
		nvt_rx_dma_timeout_get_data(nvt_port);
	}
}

static void nvt_rx_timer_init(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);

	timer_setup(&nvt_port->rx_timeout_timer, nvt_rx_timeout_timer_handle, 0);
	mod_timer(&nvt_port->rx_timeout_timer, jiffies + msecs_to_jiffies(CONFIG_SERIAL_NVT_TIMEOUT_TIME));
}
#endif

static irqreturn_t nvt_irq(int irq, void *dev_id)
{
	struct uart_port *port = dev_id;
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	unsigned int lsr_r, ier_r, iir_r,idle_r;
	unsigned long flags;
	unsigned int dma_sts=0, dma_int=0;

	spin_lock_irqsave(&port->lock, flags);
	serial_flow_dbg();

	lsr_r = nvt_read(port, UART_LSR_REG);  /* error bits are cleared after reading */
	ier_r = nvt_read(port, UART_IER_REG);
	iir_r = nvt_read(port, UART_IIR_REG);  /* clear interrupt status to avoid irq storm */
	idle_r =  nvt_read(port, UART_SPR_REG);

	if ((nvt_port->tx_dma_en || nvt_port->rx_dma_en) && port->line != 0) {
		dma_sts = nvt_read(port, UART_DMA_INT_STS_REG);
		dma_int = nvt_read(port, UART_DMA_INT_CTRL_REG);
		nvt_write(port, UART_DMA_INT_STS_REG, dma_sts);  /* clear dma interrupt status */
		serial_dbg("irq stage LSR(0x%x) IER(0x%x) IIR(0x%x) DMA_STS(0x%x) DMA_INT(0x%x)\n", lsr_r, ier_r, iir_r, dma_sts, dma_int);
	} else {
		serial_dbg("irq stage LSR(0x%x) IER(0x%x) IIR(0x%x)\n", lsr_r, ier_r, iir_r);
	}

	/* Handle UART rx */
	if ((lsr_r & DATA_READY_BIT) && (ier_r & RDA_INTEN_BIT) && (nvt_port->rx_dma_en == 0)) {
		nvt_handle_rx(port, lsr_r);
	} 
#if defined(RX_DMA_PINPON_EN_BIT) 
	else if ((iir_r & _UART_IIR_INT_CRT) && (ier_r & RDA_INTEN_BIT) && (nvt_port->rx_dma_en && port->line != 0)) {
		nvt_rx_dma_timeout_get_pinpon_data(nvt_port);
	}
#else
	else if (((dma_sts & TX_DMA_DONE_BIT)==0) && (idle_r & RX_IDLE_BIT) && (lsr_r & DATA_READY_BIT) && (ier_r & RDA_INTEN_BIT) && (nvt_port->rx_dma_en && port->line != 0)) {
		nvt_rx_dma_timeout_get_data(nvt_port);
	}
#endif

	/* Handle UART rx dma */
	if ((dma_sts & RX_DMA_DONE_BIT) && (dma_int & RX_DMA_DONE_INTEN_BIT) && (nvt_port->rx_dma_en && port->line != 0)) {

		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_DONE_INTEN_BIT, 0);
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_ERR_INTEN_BIT, 0);
		while (nvt_read(port, UART_RX_DMA_CTRL_REG) & RX_DMA_EN_BIT) {
			serial_dbg("wait RX_DMA_EN_BIT auto clear\n");
			;
		}

		nvt_port->lsr = lsr_r;
		tasklet_schedule(&nvt_port->rx_dma_tasklet);
		nvt_enable_rx_dma(port);
	/* Handle UART rx dma error */
	} else if ((dma_sts & RX_DMA_ERR_BIT) && (dma_int & RX_DMA_ERR_INTEN_BIT) && (nvt_port->rx_dma_en && port->line != 0)) {

		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_DONE_INTEN_BIT, 0);
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_ERR_INTEN_BIT, 0);
		nvt_masked_write(port, UART_RX_DMA_CTRL_REG, RX_DMA_EN_BIT, 0);
		while (nvt_read(port, UART_RX_DMA_CTRL_REG) & RX_DMA_EN_BIT) {
			;
		}

		nvt_port->lsr = lsr_r;
		tasklet_schedule(&nvt_port->rx_dma_tasklet);
		nvt_enable_rx_dma(port);

		if (dma_sts & DMA_OVERRUN_ERR_BIT)
			serial_err("dma overrun error!\r\n");

		if (dma_sts & DMA_PARITY_ERR_BIT)
			serial_err("dma parity error!\r\n");

		if (dma_sts & DMA_FRAMING_ERR_BIT)
			serial_dbg("dma framing error!\r\n");

		if (dma_sts & DMA_BREAK_ERR_BIT)
			serial_dbg("dma break interrupt!\r\n");
	}	
	
	/* Handle UART tx */
	if ((lsr_r & THR_EMPTY_BIT) && (ier_r & THR_EMPTY_INTEN_BIT)) {
		nvt_handle_tx(port);
	} else if (nvt_port->tx_dma_en && port->line != 0) {
		if ((dma_sts & TX_DMA_DONE_BIT) && (dma_int & TX_DMA_DONE_INTEN_BIT)) {
			nvt_masked_write(port, UART_DMA_INT_CTRL_REG, TX_DMA_DONE_INTEN_BIT, 0);
		}
	}

	/* Print UART error */
	if (lsr_r & FIFO_DATA_ERR_BIT) {
		if (lsr_r & OVERRUN_ERR_BIT)
			serial_err("overrun error!\r\n");

		if (lsr_r & PARITY_ERR_BIT)
			serial_err("parity error!\r\n");

		if (lsr_r & FRAMING_ERR_BIT)
			serial_dbg("framing error!\r\n");

		if (lsr_r & BREAK_INT_BIT)
			serial_dbg("break interrupt!\r\n");
	}

	/* Clear RBR when IIR stuck in CRT status */
	if (((iir_r & UART_IIR_INT_ID_MASK) == _UART_IIR_INT_CRT) && ((lsr_r & DATA_READY_BIT) == 0))
		nvt_read(port, UART_RBR_REG);
	
	spin_unlock_irqrestore(&port->lock, flags);

	return IRQ_HANDLED;
}

static unsigned int nvt_get_mctrl(struct uart_port *port)
{
	serial_flow_dbg();

	/*
	 * Pretend we have a Modem status reg and following bits are
	 *  always set, to satify the serial core state machine
	 *  (DSR) Data Set Ready
	 *  (CTS) Clear To Send
	 *  (CAR) Carrier Detect
	 */
	return TIOCM_DSR | TIOCM_CTS | TIOCM_CAR;
}

static void nvt_set_mctrl(struct uart_port *port, unsigned int mctrl)
{
	serial_flow_dbg();

	/* MCR not present */
}

static void nvt_break_ctl(struct uart_port *port, int break_ctl)
{
	unsigned long flags;

	serial_flow_dbg();

	spin_lock_irqsave(&port->lock, flags);

	if (break_ctl)
		nvt_masked_write(port, UART_LCR_REG, SET_BREAK_BIT, SET_BREAK_BIT);
	else
		nvt_masked_write(port, UART_LCR_REG, SET_BREAK_BIT, 0);

	spin_unlock_irqrestore(&port->lock, flags);
}

static int nvt_startup(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	int ret;

	serial_flow_dbg();

	snprintf(nvt_port->name, sizeof(nvt_port->name),
		 "nvt_serial%d", port->line);

	ret = request_irq(port->irq, nvt_irq, IRQF_TRIGGER_HIGH,
			  nvt_port->name, port);
	if (unlikely(ret))
		return ret;

	// coverity[side_effect_free]: spin_lock_init is kernel API, do not change
	spin_lock_init(&nvt_port->rx_timeout_lock);

	/*
	 * Clear the FIFO buffers and disable.
	 */
	nvt_clear_fifos(port);	

	/*
	 * Clear the interrupt registers again for luck, and clear the
	 * saved flags to avoid getting false values from polling
	 * routines or the previous session.
	 */
	nvt_read(port, UART_LSR_REG);
	nvt_read(port, UART_THR_REG);
	nvt_read(port, UART_IIR_REG);
	nvt_port->lsr_saved_flags = 0;
	nvt_port->lsr_break_flag = 0;

#if defined(NVT_FR) && defined(CONFIG_SERIAL_NVT_DMA)
	
	if (port->line != 0) {
		/* Set DMA feature */
		nvt_port->tx_dma_en = 1;
		nvt_port->rx_dma_en = 1;
	}
#endif

	/* A debug node to enable/disable DMA feature */
	if (__serial_tx_dma_en_debug == 1)
		nvt_port->tx_dma_en = 1;
	else if (__serial_tx_dma_en_debug == 0)
		nvt_port->tx_dma_en = 0;

	if (__serial_rx_dma_en_debug == 1)
		nvt_port->rx_dma_en = 1;
	else if (__serial_rx_dma_en_debug == 0)
		nvt_port->rx_dma_en = 0;

	/* Allocate DMA */
	if (nvt_port->tx_dma_en && port->line != 0) {
		struct circ_buf *xmit = &port->state->xmit;
		/* test DMA can be enable or not */
		nvt_port->tx_dma_addr = dma_map_single(port->dev, xmit->buf, UART_XMIT_SIZE, DMA_TO_DEVICE);

		if (dma_mapping_error(port->dev, nvt_port->tx_dma_addr)) {
			serial_err("allocate DMA failed, using PIO\n");
			nvt_port->tx_dma_en = 0;
		} else {
			serial_dbg("allocate tx_virt_addr(0x%lx) tx_dma_addr(0x%px)\n", (uintptr_t)xmit->buf,(void*)nvt_port->tx_dma_addr);
			dma_unmap_single(port->dev, nvt_port->tx_dma_addr, UART_XMIT_SIZE, DMA_TO_DEVICE);
		}
	}

	if (nvt_port->rx_dma_en && port->line != 0) {

		if (__serial_rx_size > 0 && __serial_rx_size < UART_XMIT_SIZE)
			nvt_port->rx_size = __serial_rx_size;
		else
			nvt_port->rx_size = UART_XMIT_SIZE;

		nvt_port->rx_virt_addr1 = dma_alloc_coherent(port->dev, nvt_port->rx_size, &nvt_port->rx_dma_addr1, GFP_KERNEL);
		nvt_port->rx_virt_addr2 = dma_alloc_coherent(port->dev, nvt_port->rx_size, &nvt_port->rx_dma_addr2, GFP_KERNEL);

		if (!nvt_port->rx_virt_addr1) {
			serial_err("allocate DMA failed, using PIO\n");
			nvt_port->rx_dma_en = 0;
		} else {
			serial_dbg("allocate rx_virt_addr1(0x%lx) rx_dma_addr1(0x%px)\n", (uintptr_t)nvt_port->rx_virt_addr1, (void*)nvt_port->rx_dma_addr1);
		}

		if (!nvt_port->rx_virt_addr2) {
			serial_err("allocate DMA failed, using PIO\n");
			nvt_port->rx_dma_en = 0;
		} else {
			serial_dbg("allocate rx_virt_addr2(0x%lx) rx_dma_addr2(0x%px)\n", (uintptr_t)nvt_port->rx_virt_addr2, (void*)nvt_port->rx_dma_addr2);
		}
		serial_dbg("nvt_port->rx_dma_addr1 = %lx \n",(uintptr_t)nvt_port->rx_dma_addr1);
		serial_dbg("nvt_port->rx_dma_addr2 = %lx \n",(uintptr_t)nvt_port->rx_dma_addr2);

		// enable rx dma
		nvt_port->pin_pon_flag = 0;
		nvt_write(port, UART_RX_DMA_ADDR_REG, nvt_port->rx_dma_addr1);
#if defined(UART_HIGH_ADDR_MASK) && defined(UART_RX_DMA_ADDR_H_REG)
		nvt_write(port, UART_RX_DMA_ADDR_H_REG, nvt_port->rx_dma_addr1 >> 32);
#endif
#if defined(RX_DMA_PINPON_EN_BIT) 
		nvt_write(port, UART_RX_DMA_ADDR2_REG, nvt_port->rx_dma_addr2);
#if defined(UART_HIGH_ADDR_MASK) && defined(UART_RX_DMA_ADDR2_H_REG)
		nvt_write(port, UART_RX_DMA_ADDR2_H_REG, nvt_port->rx_dma_addr2 >> 32);
#endif

		nvt_masked_write(port, UART_TIMEOUT_SETTING_REG, TIMEOUT_TIME_MASK, 0x03);
		nvt_masked_write(port, UART_TIMEOUT_SETTING_REG, TIMEOUT_TIME_SETTING_EN_BIT, TIMEOUT_TIME_SETTING_EN_BIT);
		nvt_masked_write(port, UART_RX_DMA_CTRL_REG, RX_DMA_PINPON_EN_BIT, RX_DMA_PINPON_EN_BIT);
#endif
		nvt_write(port, UART_RX_DMA_SIZE_REG, nvt_port->rx_size);
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_DONE_INTEN_BIT, RX_DMA_DONE_INTEN_BIT);
		nvt_masked_write(port, UART_DMA_INT_CTRL_REG, RX_DMA_ERR_INTEN_BIT, RX_DMA_ERR_INTEN_BIT);
		nvt_masked_write(port, UART_RX_DMA_CTRL_REG, RX_DMA_EN_BIT, RX_DMA_EN_BIT);
		tasklet_init(&nvt_port->rx_dma_tasklet, nvt_handle_rx_dma, (unsigned long)nvt_port);


		// enable hw timeout
		nvt_write(port, UART_IIR_REG, 0x0C);
		nvt_port->ch = kmalloc(32 * sizeof(char), GFP_KERNEL);
		tasklet_init(&nvt_port->timeout_tasklet, rx_dma_timeout, (unsigned long)nvt_port);
	#if defined(CONFIG_SERIAL_NVT_DMA_TIMEOUT) && !defined(RX_DMA_PINPON_EN_BIT)
		nvt_rx_timer_init(port);
	#endif
		
	}


	return 0;
}

static void nvt_shutdown(struct uart_port *port)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	unsigned long flags;

	serial_flow_dbg();

	spin_lock_irqsave(&port->lock, flags);
	nvt_write(port, UART_IER_REG, 0);  /* disable interrupts to avoid hanging in start_tx */
	spin_unlock_irqrestore(&port->lock, flags);

	free_irq(port->irq, port);

	/* Free DMA */
	if ((nvt_port->tx_dma_en || nvt_port->rx_dma_en) && port->line != 0) {
		if (nvt_port->tx_virt_addr) {
			serial_dbg("free tx_virt_addr(0x%lx) tx_dma_addr(0x%px)\n", (uintptr_t)nvt_port->tx_virt_addr, (void*)nvt_port->tx_dma_addr);
			dma_free_coherent(port->dev, UART_XMIT_SIZE, nvt_port->tx_virt_addr, nvt_port->tx_dma_addr);
			nvt_port->tx_virt_addr = NULL;
		}

		if (nvt_port->rx_virt_addr1) {
			serial_dbg("free rx_virt_addr1(0x%lx) rx_dma_addr1(0x%px)\n", (uintptr_t)nvt_port->rx_virt_addr1, (void*)nvt_port->rx_dma_addr1);
			dma_free_coherent(port->dev, nvt_port->rx_size, nvt_port->rx_virt_addr1, nvt_port->rx_dma_addr1);
			nvt_port->rx_virt_addr1 = NULL;
		}

		if (nvt_port->rx_virt_addr2) {
			serial_dbg("free rx_virt_addr2(0x%lx) rx_dma_addr2(0x%px)\n", (uintptr_t)nvt_port->rx_virt_addr2, (void*)nvt_port->rx_dma_addr2);
			dma_free_coherent(port->dev, nvt_port->rx_size, nvt_port->rx_virt_addr2, nvt_port->rx_dma_addr2);
			nvt_port->rx_virt_addr2 = NULL;
		}

		nvt_write(port, UART_DMA_INT_CTRL_REG, 0);
		nvt_write(port, UART_TX_DMA_CTRL_REG, 0);
		nvt_write(port, UART_RX_DMA_CTRL_REG, 0);

	}
}

static void nvt_set_baud_rate(struct uart_port *port, unsigned int baud)
{
	unsigned int PSR, DLR;
	struct nvt_port *nvt_port = UART_TO_NVT(port);

#if IS_ENABLED(CONFIG_NVT_IVOT_PLAT_NA51090)
	//NT98336 uart2 baudrate can not excess 115200
	if(port->line == 1 && baud > 115200){
		baud = 115200;
		serial_err("baudrate for uart2 can not excess 115200!\n");
		serial_err("baudrate set to 115200!\n");
	}
#endif
	switch (baud) {
	case 110:
		port->uartclk = 13720000;
		break;
	case 300:
	case 460800:
		port->uartclk = 39000000;
		break;
	case 9600:
		port->uartclk = 19200000;
		break;
	case 14400:
		port->uartclk = 32000000;
		break;
	case 19200:
	case 38400:
		port->uartclk = 44000000;
		break;
	case 380400:
		port->uartclk = 6100000;
		break;
	case 921600:
		port->uartclk = 47000000;
		break;
	default:
		port->uartclk = 48000000;
	}

	if (port->line == 0){
		port->uartclk = UART0_CLOCK_FREQ;
	}

	clk_set_rate(nvt_port->clk, port->uartclk);

	if (baud >= 9600) {
		PSR = 0x01;
	} else if (baud >= 2400 && baud < 9600) {
		PSR = 0x05;
	} else if (baud >= 1200 && baud < 2400) {
		PSR = 0x0a;
	} else if (baud >= 600 && baud < 1200) {
		PSR = 0x14;
	} else {
		PSR = 0x1f;
	}

	DLR = port->uartclk / 16 / PSR / baud;

	if (DLR > UART_DLR_MAX) {
		serial_err("baud out of MAX range error(DLR=%d)!\n", DLR);
	}
	if (DLR < UART_DLR_MIN) {
		serial_err("baud out of MIN range error(DLR=%d)!\n", DLR);
	}
}

static void nvt_set_baud_rate_reg(struct uart_port *port, unsigned int baud)
{
	unsigned int PSR, DLR;

#if IS_ENABLED(CONFIG_NVT_IVOT_PLAT_NA51090)
	//NT98336 uart2 baudrate can not excess 115200
	if(port->line == 1 && baud > 115200){
		baud = 115200;
		serial_err("baudrate for uart2 can not excess 115200!\n");
		serial_err("baudrate set to 115200!\n");
	}
#endif
	
	if (baud >= 9600) {
		PSR = 0x01;
	} else if (baud >= 2400 && baud < 9600) {
		PSR = 0x05;
	} else if (baud >= 1200 && baud < 2400) {
		PSR = 0x0a;
	} else if (baud >= 600 && baud < 1200) {
		PSR = 0x14;
	} else {
		PSR = 0x1f;
	}

	DLR = port->uartclk / 16 / PSR / baud;

	if (DLR > UART_DLR_MAX) {
		DLR = UART_DLR_MAX;
		serial_err("baud out of MAX range error!\n");
	}
	if (DLR < UART_DLR_MIN) {
		DLR = UART_DLR_MIN;
		serial_err("baud out of MIN range error!\n");
	}

	nvt_write(port, UART_DLL_REG, DLR & DLL_MASK);
	nvt_masked_write(port, UART_DLM_REG, DLM_MASK, DLR >> 8);
	nvt_write(port, UART_PSR_REG, PSR & PSR_MASK);
}

static void nvt_set_termios(struct uart_port *port, struct ktermios *termios,
			    struct ktermios *old)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
    unsigned long flags;
    unsigned int lcr_val = 0, baud;

    serial_flow_dbg();

    /* Set baud rate and divisor */
    spin_lock_irqsave(&port->lock, flags);
    if (nvt_port->baud) {
        if (tty_termios_baud_rate(termios))
            tty_termios_encode_baud_rate(termios, nvt_port->baud, nvt_port->baud);
        nvt_port->baud = 0;
    }
    baud = uart_get_baud_rate(port, termios, old, 0, port->uartclk / 16);
    spin_unlock_irqrestore(&port->lock, flags);

    nvt_set_baud_rate(port, baud);
    
    spin_lock_irqsave(&port->lock, flags);
    nvt_masked_write(port, UART_LCR_REG, DLAB_BIT, DLAB_BIT);  /* set DLAB bit to access divisor */
	nvt_set_baud_rate_reg(port, baud);

	/* Set length */
	switch (termios->c_cflag & CSIZE) {
	case CS5:
		lcr_val |= _WL_L5;
		break;
	case CS6:
		lcr_val |= _WL_L6;
		break;
	case CS7:
		lcr_val |= _WL_L7;
		break;
	case CS8:
	default:
		lcr_val |= _WL_L8;
		break;
	}

	/* Set stop bits */
	if (termios->c_cflag & CSTOPB)
		lcr_val |= STOP_BIT;

	/* Set parity */
	if (termios->c_cflag & PARENB) {
		if (termios->c_cflag & CMSPAR) {
			if (termios->c_cflag & PARODD)
				lcr_val |= _UART_PARITY_ONE << _UART_LCR_PARITY_SHIFT;
			else
				lcr_val |= _UART_PARITY_ZERO << _UART_LCR_PARITY_SHIFT;
		} else {
			if (termios->c_cflag & PARODD)
				lcr_val |= _UART_PARITY_ODD << _UART_LCR_PARITY_SHIFT;
			else
				lcr_val |= _UART_PARITY_EVEN << _UART_LCR_PARITY_SHIFT;
		}
	}
	nvt_write(port, UART_LCR_REG, lcr_val);  /* clear DLAB bit */

	/* Set rx fifo trigger level */
	nvt_write(port, UART_FCR_REG, FIFO_EN_BIT | (nvt_port->rx_trig_level << _UART_FCR_RX_TRIGGER_SHIFT));

	/* Set hardware flow control */
	if (nvt_port->hw_flowctrl) {
		termios->c_cflag |= CRTSCTS;
		nvt_port->hw_flowctrl = 0;
	}

#ifdef NVT_FR
	if (termios->c_cflag & CRTSCTS) {
		nvt_masked_write(port, UART_MCR_REG, HW_FLOW_CTRL_BIT, HW_FLOW_CTRL_BIT);
	} else {
		nvt_masked_write(port, UART_MCR_REG, HW_FLOW_CTRL_BIT, 0);
	}
#else
	if (termios->c_cflag & CRTSCTS) {
		nvt_masked_write(port, UART_IER_REG, HW_FLOW_CTRL_BIT, HW_FLOW_CTRL_BIT);
	} else {
		nvt_masked_write(port, UART_IER_REG, HW_FLOW_CTRL_BIT, 0);
	}
#endif

#ifdef NVT_FR
	/* Set rs485 feature */
	if (nvt_port->rs485_en) {
		nvt_write(port, UART_RS485_REG, ENABLE_BIT
			| (nvt_port->rs485_delay_before_send << _UART_RS485_SETUP_TIME_SHIFT)
			| (nvt_port->rs485_delay_after_send << _UART_RS485_HOLD_TIME_SHIFT));
	}
#endif

	if (nvt_port->rx_dma_en && port->line != 0) {
		/* Enanle receive timeout interrupts */
		nvt_write(port, UART_IER_REG, RDA_INTEN_BIT);
	} else {
		/* Enanle receive interrupts */
		nvt_write(port, UART_IER_REG, RLS_INTEN_BIT | RDA_INTEN_BIT);
	}

	/* Configure status bits to ignore based on termio flags */
	port->read_status_mask = OVERRUN_ERR_BIT;
	if (termios->c_iflag & INPCK)
		port->read_status_mask |= FRAMING_ERR_BIT | PARITY_ERR_BIT;
	if (termios->c_iflag & (IGNBRK | BRKINT | PARMRK))
		port->read_status_mask |= BREAK_INT_BIT;

	uart_update_timeout(port, termios->c_cflag, baud);

	serial_dbg("baud(%d) hw_flowctrl(%d) tx_dma(%d) rx_dma(%d)\n", baud, ((termios->c_cflag & CRTSCTS) ? 1 : 0), nvt_port->tx_dma_en, nvt_port->rx_dma_en);

	spin_unlock_irqrestore(&port->lock, flags);
}

static const char *nvt_type(struct uart_port *port)
{
	return "NVT";
}

static void nvt_release_port(struct uart_port *port)
{
	serial_flow_dbg();
}

static int nvt_request_port(struct uart_port *port)
{
	serial_flow_dbg();

	return 0;
}

static void nvt_config_port(struct uart_port *port, int flags)
{
	if (flags & UART_CONFIG_TYPE)
		port->type = PORT_16550A;
}

static int nvt_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	serial_flow_dbg();

	if (unlikely(ser->type != PORT_UNKNOWN && ser->type != PORT_16550A))
		return -EINVAL;
	if (unlikely(port->irq != ser->irq))
		return -EINVAL;
	return 0;
}

static inline void wait_for_xmitr(struct uart_port *port, int bits)
{
	struct nvt_port *nvt_port = UART_TO_NVT(port);
	unsigned int status, tmout = 10000;

	/* Wait up to 10ms for the character(s) to be sent */
	for (;;) {
		status = nvt_read(port, UART_LSR_REG);

		nvt_port->lsr_saved_flags |= status & LSR_SAVE_FLAGS;

		if (status & BREAK_INT_BIT)
			nvt_port->lsr_break_flag = BREAK_INT_BIT;

		if ((status & bits) == bits)
			break;
		if (--tmout == 0)
			break;
		udelay(1);
	}
}

#ifdef CONFIG_CONSOLE_POLL
static int nvt_poll_get_char(struct uart_port *port)
{
	unsigned char ch;

	while (!(nvt_read(port, UART_LSR_REG) & DATA_READY_BIT))
		cpu_relax();

	ch = nvt_read(port, UART_RBR_REG);

	return ch;
}

static void nvt_poll_put_char(struct uart_port *port, unsigned char ch)
{
	unsigned int ier;

	/* First save the IER then disable the interrupts */
	ier = nvt_read(port, UART_IER_REG);
	nvt_write(port, UART_IER_REG, 0);
	wait_for_xmitr(port, BOTH_EMPTY_BIT);

	/* Send the character out */
	nvt_write(port, UART_THR_REG, ch);

	/* Finally, wait for transmitter to become empty and restore the IER */
	wait_for_xmitr(port, BOTH_EMPTY_BIT);
	nvt_write(port, UART_IER_REG, ier);
}
#endif

static struct uart_ops nvt_uart_pops = {
	.stop_rx = nvt_stop_rx,
	.stop_tx = nvt_stop_tx,
	.start_tx = nvt_start_tx,
	.tx_empty = nvt_tx_empty,
	.get_mctrl = nvt_get_mctrl,
	.set_mctrl = nvt_set_mctrl,
	.break_ctl = nvt_break_ctl,
	.startup = nvt_startup,
	.shutdown = nvt_shutdown,
	.set_termios = nvt_set_termios,
	.type = nvt_type,
	.release_port = nvt_release_port,
	.request_port = nvt_request_port,
	.config_port = nvt_config_port,
	.verify_port = nvt_verify_port,
#ifdef CONFIG_CONSOLE_POLL
	.poll_get_char	= nvt_poll_get_char,
	.poll_put_char	= nvt_poll_put_char,
#endif
};

static struct nvt_port nvt_uart_ports[] = {
#ifdef CONFIG_SERIAL_NVT_CONSOLE
	{
		.uart = {
			.line = 0,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 24000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 115200, /* set into termios structure when first used, and can be modified through user interface such as stty */
		.tx_loadsz = 64, /* maximum number of characters tx can send in one handle_tx cycle, usually following fifosize */
		.hw_flowctrl = 0, /* default hardware flow control is on or off, also we can use stty crtscts/-crtscts to turn on/off */
		.rx_trig_level = 3,
	},
#endif
	{
		.uart = {
			.line = 1,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 115200,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 2,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 3,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 4,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 5,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 6,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 7,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
	{
		.uart = {
			.line = 8,
			.ops = &nvt_uart_pops,
			.flags = UPF_BOOT_AUTOCONF,
		/**********************************************************************
		 *  Following items are default values and will be overwritten by dts *
		 **********************************************************************/
			.uartclk = 48000000,
			.iotype = UPIO_MEM32,
			.fifosize = 64,
			.regshift = 2,
		},
		.baud = 0,
		.tx_loadsz = 64,
		.hw_flowctrl = 0,
		.rx_trig_level = 3,
	},
};

static inline struct uart_port *get_port_from_line(unsigned int line)
{
	return &nvt_uart_ports[line].uart;
}

#ifdef CONFIG_SERIAL_NVT_CONSOLE
static void nvt_console_putchar(struct uart_port *port, int ch)
{
	wait_for_xmitr(port, THR_EMPTY_BIT);
	nvt_write(port, UART_THR_REG, ch);
}

static void nvt_console_write(struct console *co, const char *s, unsigned int count)
{
	struct uart_port *port = get_port_from_line(co->index);
	unsigned long flags;
	unsigned int ier;
	int i;
	int locked = 1;
	
	for (i = 0; i < count; i++) {
		if (port->sysrq)
			locked = 0;
		else if (oops_in_progress)
			locked = spin_trylock_irqsave(&port->lock, flags);
		else
			spin_lock_irqsave(&port->lock, flags);

		/* First save the IER then disable the interrupts */
		ier = nvt_read(port, UART_IER_REG);
		nvt_write(port, UART_IER_REG, 0);

		uart_console_write(port, &s[i], 1, nvt_console_putchar);

		/* Finally, wait for transmitter to become empty and restore the IER */
		wait_for_xmitr(port, BOTH_EMPTY_BIT);
		nvt_write(port, UART_IER_REG, ier);

		if (locked)
			spin_unlock_irqrestore(&port->lock, flags);
	}
}

static int __init nvt_console_setup(struct console *co, char *options)
{
	struct uart_port *port;
	int baud = 115200;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	if (unlikely(co->index >= nr_uarts || co->index < 0))
		return -ENXIO;

	/*
	 * The uart port backing the console (e.g. ttyS0) might not have been
	 * init yet. If so, defer the console setup to after the port.
	 */
	port = get_port_from_line(co->index);
	if (unlikely(!port->membase))
		return -ENXIO;

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);

	pr_info("nvt_serial: console setup on port #%d\n", port->line);

	/*
	 * Serial core will call port->ops->set_termios( )
	 * which will set the baud reg.
	 */
	return uart_set_options(port, co, baud, parity, bits, flow);
}

static void nvt_serial_early_putchar(struct uart_port *port, int ch)
{
	wait_for_xmitr(port, THR_EMPTY_BIT);
	nvt_write_early(port, UART_THR_REG, ch);
}

static void nvt_serial_early_write(struct console *con, const char *s, unsigned n)
{
	struct earlycon_device *dev = con->data;

	uart_console_write(&dev->port, s, n, nvt_serial_early_putchar);
}

static int __init nvt_serial_early_console_setup(struct earlycon_device *device, const char *opt)
{
	unsigned int PSR, DLR;
	struct uart_port *port = &device->port;

	if (!device->port.membase)
		return -ENODEV;
	
	if (device->baud == 0) {
		device->baud = 115200;
		port->uartclk = UART0_CLOCK_FREQ;
	}

	//set buadrate
	PSR = 1;
	DLR = port->uartclk / 16 / PSR / device->baud;
	writel(DLAB_BIT, port->membase + UART_LCR_REG);
	writel(PSR, port->membase + UART_PSR_REG);
	writel(DLR, port->membase + UART_DLL_REG);
	writel(DLAB_BIT, port->membase + UART_LCR_REG);
	writel(_WL_L8, port->membase + UART_LCR_REG);

	//clear fifo
	writel(FIFO_EN_BIT | RX_FIFO_RESET_BIT | TX_FIFO_RESET_BIT, port->membase + UART_FCR_REG);

	device->con->write = nvt_serial_early_write;
	return 0;
}
EARLYCON_DECLARE(nvt_serial, nvt_serial_early_console_setup);
OF_EARLYCON_DECLARE(nvt_serial, "ns16550a", nvt_serial_early_console_setup);

static struct uart_driver nvt_uart_driver;

static struct console nvt_console = {
	.name = "ttyS",
	.write = nvt_console_write,
	.device = uart_console_device,
	.setup = nvt_console_setup,
	.flags = CON_PRINTBUFFER,
	.index = -1,
	.data = &nvt_uart_driver,
};

#define NVT_CONSOLE	(&nvt_console)
#else
#define NVT_CONSOLE	NULL
#endif

#ifdef AB_MODIFIED
ssize_t ab_logdump_read(struct file *filp, char __user *user_buffer, size_t size, loff_t *offset)
{
    AB_LOGDUMP_OPEN_INFO_ST *ps_dev = filp->private_data;
    int32_t i_tar_size = 0, i_ret_size = 0;
	unsigned long comm_flags = 0;
#if AB_LOGDUMP_EN_TIME_MEASUREMENT
    bool b_en_timerec = false;
    static bool b_en_calculate = false;
#endif

    if(ps_dev == NULL) return -EFAULT;

    // check if there have any data need to pass to userspace
    if(iab_logdump_queue_wait_count() <= 0) {
        if(filp->f_flags & O_NONBLOCK) {
            // skip wait event
            return 0;
        }
        else {
			unsigned long flags = 0;
			spin_lock_irqsave(&ps_open_info->s_read_lock, flags);
            //down(&ps_dev->s_protect_sem);
            ps_dev->b_wait_wakeup = true;
            //up(&ps_dev->s_protect_sem);
            spin_unlock_irqrestore(&ps_open_info->s_read_lock, flags);
        #if AB_LOGDUMP_EN_TIME_MEASUREMENT
            b_en_timerec = true;
            _get_boottime(&ps_desc->s_measurement[0]);
            if(b_en_calculate) {
                int32_t i_val = (int32_t)((ps_desc->s_measurement[0].tv_sec - ps_desc->s_measurement[3].tv_sec) * 1000000) + (int32_t)((ps_desc->s_measurement[0].tv_nsec-ps_desc->s_measurement[3].tv_nsec)/1000);
                if(i_val > ps_desc->ia_measurement[0])
                    ps_desc->ia_measurement[0] = i_val;
            }
            wait_event_interruptible(ps_dev->t_read_wait, ps_dev->b_wait_wakeup != true);
            _get_boottime(&ps_desc->s_measurement[2]);
            if(b_en_calculate) {
                int32_t i_val = 0;
                i_val = (int32_t)((ps_desc->s_measurement[1].tv_sec - ps_desc->s_measurement[0].tv_sec) * 1000000) + (int32_t)((ps_desc->s_measurement[1].tv_nsec-ps_desc->s_measurement[0].tv_nsec)/1000);

                if(i_val > ps_desc->ia_measurement[1])
                    ps_desc->ia_measurement[1] = i_val;

                i_val = (int32_t)((ps_desc->s_measurement[2].tv_sec - ps_desc->s_measurement[1].tv_sec) * 1000000) + (int32_t)((ps_desc->s_measurement[2].tv_nsec-ps_desc->s_measurement[1].tv_nsec)/1000);
                if(i_val > ps_desc->ia_measurement[2])
                    ps_desc->ia_measurement[2] = i_val;
            }
        #else
            AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "%s wait_event_interruptible s\n", __func__);
			#if AB_LOGDUMP_EN_BLOCK_READ_TIMEOUT
            wait_event_interruptible_timeout(ps_dev->t_read_wait, ps_dev->b_wait_wakeup != true, AB_LOGDUMP_WAKEUP_TIMEOUT_JIFFIES);
			#else
            wait_event_interruptible(ps_dev->t_read_wait, ps_dev->b_wait_wakeup != true);
			#endif
            AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "%s wait_event_interruptible e\n", __func__);
        #endif
        }
    }   
#if AB_LOGDUMP_EN_TIME_MEASUREMENT
    else {
        b_en_calculate = false;
        b_en_timerec = true;
        _get_boottime(&ps_desc->s_measurement[0]);
        _get_boottime(&ps_desc->s_measurement[1]);
        _get_boottime(&ps_desc->s_measurement[2]);
    }
#endif

    i_tar_size = (int32_t) size;

    while(i_tar_size >= AB_LOGDUMP_BUFF_DATA_MAXSIZE) {
        if(iab_logdump_queue_wait_count() > 0) {
            AB_LOGDUMP_BUFF_ST* ps_target_logbuff = NULL;

            if(iab_logdump_pop_queue(&ps_target_logbuff) != 0) {
                // pop fail, it may be overwrited
                break;
            }

            if(ps_target_logbuff->uw_datalen > AB_LOGDUMP_BUFF_DATA_MAXSIZE) {
                AB_LOGDUMP_PRINT(true, "%s cur buff data len %d over limit %d \n", __FUNCTION__, ps_target_logbuff->uw_datalen, AB_LOGDUMP_BUFF_DATA_MAXSIZE);
                break;
            }

            if (copy_to_user(user_buffer + i_ret_size, ps_target_logbuff->uc_data, ps_target_logbuff->uw_datalen))
                return -EFAULT;


            i_ret_size  += (int32_t)ps_target_logbuff->uw_datalen;
            i_tar_size  -= (int32_t)ps_target_logbuff->uw_datalen;
spin_lock_irqsave(&ab_logdump_lock, comm_flags);
            memset(ps_target_logbuff->uc_data, 0, sizeof(ps_target_logbuff->uc_data));
            ps_target_logbuff->uc_inqueue   = false;
            ps_target_logbuff->uw_datalen   = 0;
            ps_target_logbuff->uc_bufstatus = AB_LOGDUMP_BUFF_VALID;
spin_unlock_irqrestore(&ab_logdump_lock, comm_flags);            

        }
        else {
            break;
        }
    }

#if AB_LOGDUMP_EN_TIME_MEASUREMENT
    if(b_en_timerec) {
        _get_boottime(&ps_desc->s_measurement[3]);
        b_en_calculate = true;
        if(b_en_calculate) {
            int32_t i_val = (int32_t)((ps_desc->s_measurement[3].tv_sec - ps_desc->s_measurement[2].tv_sec) * 1000000) + (int32_t)((ps_desc->s_measurement[3].tv_nsec-ps_desc->s_measurement[2].tv_nsec)/1000);
            if(i_val > ps_desc->ia_measurement[3]) 
                ps_desc->ia_measurement[3] = i_val;
        }
    }
#endif

    return i_ret_size;
}

long ab_logdump_ioctl(struct file *filp, unsigned int cmd, unsigned long args)
{
	AB_LOGDUMP_OPEN_INFO_ST *ps_dev = filp->private_data;

    if(_IOC_TYPE(cmd) != IOC_MAGIC) {
        return -ENOTTY;
    }

    if(_IOC_NR(cmd) > IOC_MAX_NUMBER) {
        return -ENOTTY;
    }

    if(ps_desc == NULL || ps_dev == NULL) {
        return -ENOTTY;
    }

	switch (cmd) {

		case AB_FEATURE_CTRL: {
            AB_LOGDUMP_CTRL_DATA_ST s_ctrl_data = {0};

			//copy flag setting from user(args) to kernel(s_ctrl_data)
			if (copy_from_user(&s_ctrl_data, (uint8_t __user *)args, sizeof(AB_LOGDUMP_CTRL_DATA_ST))) {
				return -1;
			}

			switch (s_ctrl_data.uc_feature_idx) {
				case AB_LOGDUMP_CTRL_FEATURE_UART_INIT: {
					ps_desc->b_uartinit_ready = true;
                    AB_LOGDUMP_PRINT(true, "[ab_log] uart init\n");
					break;
                }
                    
				case AB_LOGDUMP_CTRL_FEATURE_FORCE_DUMP_LOG: {
                    ps_desc->b_logdump_forcedump_trig = true;
                    AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "[ab_log] trig dump log\n");
					break;
                }

				case AB_LOGDUMP_CTRL_FEATURE_TERMINATE: {
					ps_desc->b_logdump_terminate_trig = true;
                    AB_LOGDUMP_PRINT(true, "[ab_log] trig terminate\n");
					break;
                }
                    
				case AB_LOGDUMP_CTRL_FEATURE_TIMESTAMP_UNIT: {
                    // AB_LODDUMP_TIMESTAMP_ACCURACY_MS AB_LODDUMP_TIMESTAMP_ACCURACY_US
                    ps_desc->e_timstamp_accuracy = s_ctrl_data.uc_feature_data;
                    AB_LOGDUMP_PRINT(true, "[ab_log] timestamp accuracy %s\n", (ps_desc->e_timstamp_accuracy == AB_LODDUMP_TIMESTAMP_ACCURACY_MS) ? "MS" : "US");
					break;
                }

                case AB_LOGDUMP_CTRL_FEATURE_EN_DEBUG_LOG: {
                    if(ps_desc->b_logdump_en_debuglog == false)
                        ps_desc->b_logdump_en_debuglog = true;
                    else 
                        ps_desc->b_logdump_en_debuglog = false;
                    AB_LOGDUMP_PRINT(true, "[ab_log] %s debug log\n", (ps_desc->b_logdump_en_debuglog) ? "enable" : "disable");
                    break;
                }

                case AB_LOGDUMP_CTRL_FEATURE_DUMP_MEASURE_OUTPUT: {
                    #if AB_LOGDUMP_EN_TIME_MEASUREMENT
                    int32_t i_ret = 0;
                    i_ret = (ps_desc->ia_measurement[0]/1000);
                    AB_LOGDUMP_PRINT(true, "[ab_log] userspace max proc time :%d ms\n", i_ret);
                    i_ret = (ps_desc->ia_measurement[1]/1000);
                    AB_LOGDUMP_PRINT(true, "[ab_log] kernel fill data time :%d ms\n", i_ret);
                    i_ret = (ps_desc->ia_measurement[2]);
                    AB_LOGDUMP_PRINT(true, "[ab_log] kernel wakeup time :%d us\n", i_ret);
                    i_ret = (ps_desc->ia_measurement[3]);
                    AB_LOGDUMP_PRINT(true, "[ab_log] kernel copy_to_userspace time :%d us\n", i_ret);
                    #endif
                    break;
                }

				default:
					break;
            }
			break;
        }

		default:  //redundant. as cmd was checked against MAXNR
			return -ENOTTY;

	}
	return 0;
}

#if AB_LOGDUMP_EN_POLL_FEATURE
unsigned int ab_logdump_poll(struct file *filp, poll_table *wait)
{
    AB_LOGDUMP_OPEN_INFO_ST *ps_dev = filp->private_data;
    unsigned int mask = 0;//POLLOUT | POLLWRNORM;

    if (ps_dev == NULL) {
        return -EBADFD;
    }

    down(&ps_dev->s_protect_sem);
    poll_wait(filp, &ps_dev->t_read_wait, wait);
    if (iab_logdump_queue_wait_count() > 0) {
        mask |= POLLIN | POLLRDNORM;
    }
    up(&ps_dev->s_protect_sem);

    AB_LOGDUMP_PRINT(ps_desc->b_logdump_en_debuglog, "%s (%d)\n", __func__, mask);

    return (mask);
}

#endif

static int ab_logdump_open(struct inode *inode, struct file *filp)
{
    if(ps_open_info == NULL) {
        ps_open_info = (AB_LOGDUMP_OPEN_INFO_ST*)kzalloc(sizeof(AB_LOGDUMP_OPEN_INFO_ST), GFP_KERNEL);
        if (ps_open_info == NULL) {
            AB_LOGDUMP_PRINT(true, "%s: not enough memory to alloc AB_LOGDUMP_OPEN_INFO_ST\n", __func__);
            return -ENOMEM;
        }

        init_waitqueue_head(&ps_open_info->t_read_wait);
		spin_lock_init(&ps_open_info->s_read_lock);
        init_MUTEX(&ps_open_info->s_protect_sem);
        ps_open_info->b_wait_wakeup = false;
    }
    filp->private_data = ps_open_info;

    AB_LOGDUMP_PRINT(true, "%s: major %d minor %d (pid %d)\n", __func__, imajor(inode), iminor(inode), current->pid);
    return 0;
}

static int ab_logdump_close(struct inode *inode, struct file *filp)
{
    AB_LOGDUMP_OPEN_INFO_ST *ps_dev = filp->private_data;

    if (ps_dev) {
        /*kfree(ps_dev);*/
        filp->private_data = NULL;
    }

    if(ps_desc != NULL) {
        ps_desc->b_uartinit_ready = false;
        ps_desc->b_logdump_terminate_trig = false;
        ps_open_info->b_wait_wakeup = false;
    }


    AB_LOGDUMP_PRINT(true, "%s: major %d minor %d (pid %d)\n", __func__, imajor(inode), iminor(inode), current->pid);
    return 0;
}

struct file_operations s_logdump_fops = {
    .open           = ab_logdump_open,
    .release        = ab_logdump_close,
    .read           = ab_logdump_read,
    .unlocked_ioctl = ab_logdump_ioctl,
#if AB_LOGDUMP_EN_POLL_FEATURE
    .poll           = ab_logdump_poll,
#endif
};

static int32_t ab_logdump_deinit(void)
{
    if(ps_desc) {
        // deinit device and class
        device_destroy(ps_desc->ps_class, ps_desc->ui_logdump_dev);
        class_destroy(ps_desc->ps_class);
    
        // deinit cdev and region
        if(ps_desc->b_cdev_ready == true) {
            cdev_del(&ps_desc->ps_cdev);
            ps_desc->b_cdev_ready = false;
        }

        if(ps_desc->b_dev_region_ready == true) {
            unregister_chrdev_region(ps_desc->ui_logdump_dev , i_devs);
            ps_desc->b_dev_region_ready = false;
        }

        // free descrption
        if(ps_desc) {
            kfree(ps_desc);
            ps_desc = NULL;
        }
    }
    
#if AB_LOGDUMP_EN_KFIFO
    if(kfifo_initialized(&s_transfer_queue) != false) {
        kfifo_free(&s_transfer_queue);
        memset(&s_transfer_queue, 0, sizeof(s_transfer_queue));
    }
#endif

    // free main/dynamic buffer pool
    if(pv_main_buff_pool) {
        kfree(pv_main_buff_pool);
        pv_main_buff_pool = NULL;
    }
    
    if(pv_dynamic_buff_pool) {
        kfree(pv_dynamic_buff_pool);
        pv_dynamic_buff_pool = NULL;
    }

    ps_logbuff = NULL;
    ps_curlogbuff = NULL;

#if !AB_LOGDUMP_EN_KFIFO
    memset(ps_transfer_queue, 0, sizeof(ps_transfer_queue));
    memset(&s_transfer_info, 0, sizeof(s_transfer_info));
#endif
    memset(&s_buff_info, 0, sizeof(s_buff_info));

    return 0;
}

static int32_t ab_logdump_init(void)
{
    uint8_t uc_idx = 0;
    AB_LOGDUMP_BUFF_ST* ps_curr_pool = NULL;
    AB_LOGDUMP_BUFF_ST* ps_curr_buff = NULL;

    // alloc description information of log dump service
    if(ps_desc == NULL) {
        ps_desc = kzalloc(sizeof(AB_LOGDUMP_DESC_INFO_ST), GFP_KERNEL);

        if(ps_desc == NULL) {
            AB_LOGDUMP_PRINT(true, "%s: kzalloc description information failed.\n", __FUNCTION__);
            ab_logdump_deinit();
            return -1;            
        }
    }

    // alloc main/dynamic buffer pool
    if(pv_main_buff_pool == NULL) {
        pv_main_buff_pool = kzalloc(AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM*sizeof(AB_LOGDUMP_BUFF_ST), GFP_KERNEL);

        if(pv_main_buff_pool == NULL) {
            AB_LOGDUMP_PRINT(true, "%s: kzalloc main msgbuf pool failed.\n", __FUNCTION__);
            ab_logdump_deinit();
            return -1;
        }
    }

    if(pv_dynamic_buff_pool == NULL) {
        pv_dynamic_buff_pool = kzalloc(AB_LOGDUMP_DYNAMIC_POOL_BUFF_MAX_NUM*sizeof(AB_LOGDUMP_BUFF_ST), GFP_KERNEL);

        if (pv_dynamic_buff_pool == NULL) {
            AB_LOGDUMP_PRINT(true, "%s: kzalloc dynamic msgbuf pool failed.\n", __FUNCTION__);
            ab_logdump_deinit();
            return -1;
        }
    }

#if AB_LOGDUMP_EN_KFIFO
    if(kfifo_initialized(&s_transfer_queue) == false) {
        int i_ret = 0;
        i_ret = kfifo_alloc(&s_transfer_queue, (AB_LOGDUMP_BUFF_MAX_CNT * sizeof(AB_LOGDUMP_BUFF_ST*)), GFP_KERNEL);

        if(i_ret != 0) {
            ab_logdump_deinit();
            AB_LOGDUMP_PRINT(true, "%s:%d kfifo init & alloc fail\n", __FUNCTION__, __LINE__);
            return -1;
        }
    }
#endif

    if(ps_desc->b_logdump_buff_ready == false) {
        // separate to multi-buffer with structure format AB_LOGDUMP_BUFF_ST
        ps_curr_pool = (AB_LOGDUMP_BUFF_ST*)pv_main_buff_pool;
        for(uc_idx = 0; uc_idx<AB_LOGDUMP_BUFF_MAX_CNT; uc_idx++) {
            if(uc_idx < AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM) { // handle main buffer
                // first case sould fill addr to ps_logbuff
                if(uc_idx == 0) {
                    ps_logbuff = &ps_curr_pool[uc_idx];
                    ps_curr_buff = ps_logbuff;
                }

                // decide next buffer addr
                if(uc_idx == (AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM-1)) {
                    ps_curr_pool = (AB_LOGDUMP_BUFF_ST*)pv_dynamic_buff_pool;
                    //ps_curr_buff->ul_next_bufptr = (uint64_t)&ps_curr_pool[0];
					ps_curr_buff->ul_next_bufptr = &ps_curr_pool[0];
                }
                else {
                    //ps_curr_buff->ul_next_bufptr = (uint64_t)&ps_curr_pool[uc_idx+1];
					ps_curr_buff->ul_next_bufptr = &ps_curr_pool[uc_idx+1];
                }
            }
            else {  // handle dynamic buffer
                uint8_t uc_dynmaic_buf_idx = uc_idx - AB_LOGDUMP_MAIN_POOL_BUFF_MAX_NUM;
                // decide next buffer addr
                if(uc_idx == (AB_LOGDUMP_BUFF_MAX_CNT-1)) {
                    // last buffer need to special handle for loop link buffer
                    //ps_curr_buff->ul_next_bufptr = (uint64_t)ps_logbuff;
					ps_curr_buff->ul_next_bufptr = ps_logbuff;
                }
                else {
                    //ps_curr_buff->ul_next_bufptr = (uint64_t)&ps_curr_pool[uc_dynmaic_buf_idx+1];
					ps_curr_buff->ul_next_bufptr = &ps_curr_pool[uc_dynmaic_buf_idx+1];
                }
            }
            ps_curr_buff = (AB_LOGDUMP_BUFF_ST*)ps_curr_buff->ul_next_bufptr;
        }

        _get_boottime(&ps_desc->s_ts);

        ps_curlogbuff = ps_logbuff;
        ps_curlogbuff->uc_bufstatus = AB_LOGDUMP_BUFF_IN_USE;
        s_buff_info.uc_buff_max_idx = AB_LOGDUMP_BUFF_MAX_CNT;
        s_buff_info.b_is_occur_linebreak = false;
        s_buff_info.b_logbuff_en_overwrite = false;
        s_buff_info.ul_drop_log_len = 0;
        s_buff_info.ui_overwrite_buff_cnt = 0;
    #if !AB_LOGDUMP_EN_KFIFO    
        s_transfer_info.c_input_idx = -1;
        s_transfer_info.c_output_idx = -1;
    #endif
        ps_desc->b_logdump_buff_ready = true;
    }

    // create character device
    if(ps_desc->b_logdump_dev_ready == false) {
        int32_t i_ret = 0;
        
        ps_desc->ui_logdump_dev = MKDEV(i_major, i_minor);
        i_ret = alloc_chrdev_region(&ps_desc->ui_logdump_dev , 0, i_devs, AB_LOGDUMP_DEV_NAME);

        if(i_ret) {
            AB_LOGDUMP_PRINT(true, "%s: logdump driver cdev add failed\n", __FUNCTION__);
            ab_logdump_deinit();
            return -1;            
        }

        ps_desc->b_dev_region_ready = true;

        i_major = MAJOR(ps_desc->ui_logdump_dev);
        cdev_init(&ps_desc->ps_cdev, &s_logdump_fops);
        ps_desc->ps_cdev.owner = THIS_MODULE;
        ps_desc->ps_cdev.ops = &s_logdump_fops;

        i_ret = cdev_add(&ps_desc->ps_cdev, MKDEV(i_major, i_minor), 1);
        if (i_ret) {
            AB_LOGDUMP_PRINT(true, "%s: logdump uc driver cdev add failed\n", __FUNCTION__);
            ab_logdump_deinit();
            return -1;
        }

        ps_desc->b_cdev_ready = true;

        ps_desc->ps_class = class_create(THIS_MODULE, AB_LOGDUMP_DEV_NAME);
        if (IS_ERR(ps_desc->ps_class)) {
            AB_LOGDUMP_PRINT(true, "%s: logdumpuc uart driver create class failed\n", __FUNCTION__);
            ab_logdump_deinit();
            return -1;
        }
        
        ps_desc->ps_device = device_create(ps_desc->ps_class, NULL, ps_desc->ui_logdump_dev , NULL, AB_LOGDUMP_DEV_NAME);
        AB_LOGDUMP_PRINT(true, "%s: logdumpuc uart driver(major %d) character device create successfully\n", __FUNCTION__, i_major);
        
        ps_desc->b_logdump_dev_ready = true;
    }

    return 0;
}

#endif

static struct uart_driver nvt_uart_driver = {
	.owner = THIS_MODULE,
	.driver_name = DRIVER_NAME,
	.dev_name = "ttyS",
	.cons = NVT_CONSOLE,
};

static int nvt_serial_probe(struct platform_device *pdev)
{
	struct nvt_port *nvt_port;
	struct uart_port *port;
	u32 prop, prop_array[2] = {0};
	int line, irq;
	struct resource *resource;

#ifdef AB_MODIFIED
    if(ab_logdump_init()) {
        AB_LOGDUMP_PRINT(true, "%s: init logdump service fail\n", __FUNCTION__);
    }
#endif


	line = of_alias_get_id(pdev->dev.of_node, "serial");
	if (line < 0) {
		dev_dbg(&pdev->dev, "get line failed, use default line(%d)\n", global_line);
		line = global_line;
		global_line++;
	}

	if (line >= nr_uarts) {
		dev_err(&pdev->dev, "unsupported uart line(%d)\n", line);
		return -ENXIO;
	}

#if defined(CONFIG_NVT_IVOT_PLAT_NA51055)
	nvt_uart_ports[0].uart.fifosize = 16;
	nvt_uart_ports[0].tx_loadsz = 16;
#endif

	port = get_port_from_line(line);
	port->dev = &pdev->dev;
	nvt_port = UART_TO_NVT(port);

	// coverity[side_effect_free]: spin_lock_init is kernel API, do not change
	spin_lock_init(&nvt_port->write_lock);
	
	resource = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (unlikely(!resource)) {
		dev_err(&pdev->dev, "get address failed\n");
		return -ENXIO;
	}
	port->mapbase = resource->start;

	port->membase = devm_ioremap(port->dev, resource->start, resource_size(resource));
	if (!port->membase) {
		dev_err(&pdev->dev, "get ioremap failed\n");
		return -EBUSY;
	}

	irq = irq_of_parse_and_map(pdev->dev.of_node, 0);
	if (unlikely(irq < 0)) {
		dev_err(&pdev->dev, "get irq failed\n");
		return -ENXIO;
	}
	port->irq = irq;
	port->has_sysrq = IS_ENABLED(CONFIG_SERIAL_NVT_CONSOLE);

	if (of_property_read_u32(pdev->dev.of_node, "reg-shift", &prop) == 0)
		port->regshift = prop;

	if (of_property_read_u32(pdev->dev.of_node, "reg-io-width", &prop) == 0) {
		switch (prop) {
		case 1:
			port->iotype = UPIO_MEM;
			break;
		case 4:
			port->iotype = of_device_is_big_endian(pdev->dev.of_node) ?
				       UPIO_MEM32BE : UPIO_MEM32;
			break;
		default:
			dev_warn(&pdev->dev, "unsupported reg-io-width(%d)\n", prop);
		}
	}

	if (of_find_property(pdev->dev.of_node, "no-loopback-test", NULL))
		port->flags |= UPF_SKIP_TEST;

	if (of_property_read_u32(pdev->dev.of_node, "clock-frequency", &prop) == 0) {
		port->uartclk = prop;
	} else {
		dev_dbg(&pdev->dev, "get clock-frequency failed, use default uartclk(%d)\n", port->uartclk);
	}

	nvt_port->clk = clk_get(&pdev->dev, dev_name(&pdev->dev));
	if (IS_ERR(nvt_port->clk)) {
		dev_err(&pdev->dev, "clk %s not found\n", dev_name(&pdev->dev));
		return PTR_ERR(nvt_port->clk);
	} else {
		if (!__clk_is_enabled(nvt_port->clk))
			clk_prepare_enable(nvt_port->clk);

		clk_set_rate(nvt_port->clk, port->uartclk);
	}

	if (of_property_read_u32(pdev->dev.of_node, "baud", &prop) == 0)
		nvt_port->baud = prop;

	if (of_property_read_u32(pdev->dev.of_node, "rx_trig_level", &prop) == 0)
		nvt_port->rx_trig_level = prop;

	if (of_property_read_u32(pdev->dev.of_node, "hw_flowctrl", &prop) == 0)
		nvt_port->hw_flowctrl = prop;

	if (of_property_read_u32(pdev->dev.of_node, "rs485_en", &prop) == 0) {
		nvt_port->rs485_en = prop;

		if (of_property_read_u32_array(pdev->dev.of_node, "rs485_delay", prop_array, 2) == 0) {
			nvt_port->rs485_delay_before_send = prop_array[0];
			nvt_port->rs485_delay_after_send = prop_array[1];
		}
	}

	serial_dbg("driver probed\n");

	platform_set_drvdata(pdev, port);

	return uart_add_one_port(&nvt_uart_driver, port);
}

static int nvt_serial_remove(struct platform_device *pdev)
{
	struct uart_port *port = platform_get_drvdata(pdev);
#ifdef AB_MODIFIED
    ab_logdump_deinit();
#endif
	uart_remove_one_port(&nvt_uart_driver, port);

	return 0;
}

static const struct of_device_id nvt_match_table[] = {
	{ .compatible = "nvt,nvt_serial" },
	{ .compatible = "ns16550a" },
	{},
};

#ifdef CONFIG_PM
static int nvt_uart_suspend(struct device *dev)
{
	struct nvt_port *nvt_port;
	struct uart_port *port = dev_get_drvdata(dev);

	nvt_port = UART_TO_NVT(port);

	uart_suspend_port(&nvt_uart_driver, port);

	return 0;
}

static int nvt_uart_resume(struct device *dev)
{
	struct nvt_port *nvt_port;
	struct uart_port *port = dev_get_drvdata(dev);

	nvt_port = UART_TO_NVT(port);

	uart_resume_port(&nvt_uart_driver, port);

	return 0;
}

static const struct dev_pm_ops nvt_uart_pm_ops = {
	.suspend = nvt_uart_suspend,
	.resume = nvt_uart_resume,
};
#endif

static struct platform_driver nvt_serial_platform_driver = {
	.probe		= nvt_serial_probe,
	.remove		= nvt_serial_remove,
	.driver		= {
		.name	= DRIVER_NAME,
		.of_match_table = of_match_ptr(nvt_match_table),
#ifdef CONFIG_PM
		.pm = &nvt_uart_pm_ops,
#endif
	},
};

static int __init nvt_serial_init(void)
{
	int ret;

#ifdef CONFIG_NVT_PCIE_LIB
	nr_uarts = nr_uarts + (nr_uarts * nvtpcie_get_ep_count());
#endif

	nvt_uart_driver.nr = nr_uarts;

	ret = uart_register_driver(&nvt_uart_driver);
	if (unlikely(ret))
		return ret;

	ret = platform_driver_register(&nvt_serial_platform_driver);
	if (unlikely(ret)) {
		uart_unregister_driver(&nvt_uart_driver);
		return ret;
	}

	pr_info("%s: driver initialized, nr_uarts = %d\n", DRIVER_NAME, nr_uarts);

	return ret;
}

static void __exit nvt_serial_exit(void)
{
	platform_driver_unregister(&nvt_serial_platform_driver);
	uart_unregister_driver(&nvt_uart_driver);
}

module_init(nvt_serial_init);
module_exit(nvt_serial_exit);

MODULE_AUTHOR("Shawn Chou <shawn_chou@novatek.com.tw>");
MODULE_DESCRIPTION("Driver for NVT Soc");
MODULE_VERSION(DRIVER_VERSION);
MODULE_LICENSE("GPL");
